/**
 ****************************************************************************
 * @file     initial.c
 * @brief    Initialize Functions for TOSHIBA 'TMPM374' Device Series
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include "ipdefine.h"
#include "mcuip_drv.h"

#define DEFINE_APP
#include "initial.h"
#undef  DEFINE_APP


/*===================================================================*
	  Constant Data Definition
 *===================================================================*/
/*  */
#define	INT4KH_TB_CH			TSB_TB0
#define	INT4KH_IRQ_INTTX		INTTB00_IRQn
#define INT4KH_IRQ_LEVEL		INT4KH_LEVEL

#define	TMRB_PRD_4KH			TMRB_DIV8_PRD(250)

/**
 *********************************************************************************************
 * @brief		Initialize 4kH interval Timer.
 *
 * @return		none
 *********************************************************************************************
 */
void init_Timer_interval4kH(void)
{
	TMRB_InitTypeDef m_tmrb;

	m_tmrb.Mode = TMRB_INTERVAL_TIMER;
	m_tmrb.ClkDiv = TMRB_CLK_DIV_8;
	m_tmrb.TrailingTiming = TMRB_PRD_4KH;						/* periodic time */
	m_tmrb.UpCntCtrl = TMRB_AUTO_CLEAR;
	m_tmrb.LeadingTiming = TMRB_PRD_4KH;						/* periodic time */

	TMRB_Enable(INT4KH_TB_CH);
	TMRB_Init(INT4KH_TB_CH, &m_tmrb);

	NVIC_SetPriority(INT4KH_IRQ_INTTX, INT4KH_IRQ_LEVEL);
	NVIC_ClearPendingIRQ(INT4KH_IRQ_INTTX);
	NVIC_EnableIRQ(INT4KH_IRQ_INTTX);

	TMRB_SetRunState(INT4KH_TB_CH, TMRB_RUN);
}


/**
 *********************************************************************************************
 * @brief		Initialize ADC
 *
 * @return		none
 *********************************************************************************************
 */
void init_ADCen(void)
{
	AD_InitTypeDef adini;

/******** Setting for CH1 *********************************************/
	#define	__USE_ADUNIT_B
	adini.shunt = cSHUNT_TYPE_CH1;
	adini.iuch = cADCH_CURRENT_U_CH1;
	adini.ivch = cADCH_CURRENT_V_CH1;
	adini.iwch = cADCH_CURRENT_W_CH1;
	adini.idcch = cADCH_CURRENT_IDC_CH1;
	adini.vdcch = cADCH_VDC_CH1;
	adini.pmd_ch = cPMD1;
	adini.pints = cPINTS_B;

	IP_ADC_init(TSB_ADB, &adini);

#if defined(__MOTOR_DEBUG_INT)
	NVIC_SetPriority(INTADBPDB_IRQn, 2);
	NVIC_ClearPendingIRQ(INTADBPDB_IRQn);
	NVIC_EnableIRQ(INTADBPDB_IRQn);
#endif /* defined(__MOTOR_DEBUG) */


#if defined(__USE_ADUNIT_B)
	/* Set ADC Unit B Clock */
	ADC_SetClk(TSB_ADB, ADC_HOLD_FIX, ADC_FC_DIVIDE_LEVEL_2);
	/* Enable ADC Unit B */
	ADC_Enable(TSB_ADB);
#endif /* __USE_ADUNIT_B */

}

/**
 *********************************************************************************************
 * @brief		Initialize PMD
 *
 * @return		none
 *********************************************************************************************
 */
void init_PMDen(void)
{
	PMD_InitTypeDef	pmdini;

/******** Setting for CH1 *********************************************/
	pmdini.shunt = cSHUNT_TYPE_CH1;
	pmdini.poll = cPOLL_CH1;
	pmdini.polh = cPOLH_CH1;
	pmdini.pwmfreq = PWMPeriod(cPWMPRD_CH1);
	pmdini.deadtime = DeadTime(cDEADTIME_CH1);

	IP_PMD_init(TSB_PMD1, &pmdini);

	/* Setting interrupt. */
	NVIC_SetPriority(INTPMD1_IRQn, INT_VC_LEVEL+1);
	NVIC_ClearPendingIRQ(INTPMD1_IRQn);
	if (pmdini.shunt == 1)
	{
		NVIC_EnableIRQ(INTPMD1_IRQn);
	}
	else
	{
		NVIC_DisableIRQ(INTPMD1_IRQn);
	}

}


/**
 *********************************************************************************************
 * @brief		Initialize VE.
 *
 * @return		none
 *********************************************************************************************
 */
void init_VEen(void)
{
	VE_InitTypeDef	veini;

/******** Setting for CH1 *********************************************/
	veini.shunt = cSHUNT_TYPE_CH1;
	veini.pwmfreq = PWMPeriod(cPWMPRD_CH1);
	veini.reptime = cREPTIME_CH1;
	veini.trgmode = cTRGMODE1_INI;
	veini.tpwm = (uint16_t)(cPWMPRD_CH1 * 1e-6 * cHZ_MAX_CH1 * (1 << 16U));

	veini.idkp = (q15_t)(FIX_15 * cID_KP_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
	veini.idki = (q15_t)(FIX_15 * cID_KI_CH1 * cCTRL_PRD_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
	veini.iqkp = (q15_t)(FIX_15 * cIQ_KP_CH1 * cA_MAX_CH1 / cV_MAX_CH1);
	veini.iqki = (q15_t)(FIX_15 * cIQ_KI_CH1 * cCTRL_PRD_CH1 * cA_MAX_CH1 / cV_MAX_CH1);

	veini.zerooffset = ADC_V2HEX(cSHUNT_ZERO_OFFSET_CH1);

	veini.ve_ch = cCH1;

	NVIC_SetPriority(INTVCN1_IRQn, INT_VC_LEVEL);		/* Interruption level set */
	NVIC_ClearPendingIRQ(INTVCN1_IRQn);
	NVIC_EnableIRQ(INTVCN1_IRQn);						/* INTVCNx enable */

	IP_VE_init(TSB_VE, &veini);

}

/**
 *********************************************************************************************
 * @brief		Initialize WDT
 *
 * @return		none
 *********************************************************************************************
 */
void init_WDT(void)
{

	WDT_InitTypeDef WDTini;

	WDTini.DetectTime = WDT_DETECT_TIME_EXP_25;
	WDTini.OverflowOutput = WDT_WDOUT;

	WDT_Init(&WDTini);
	WDT_WriteClearCode();
	WDT_Enable();

}

/*********************************** END OF FILE ******************************/
