/**
 ****************************************************************************
 * @file	initial.c
 * @brief	Initialize Functions for TOSHIBA 'TMPM4K' Device Series
 * @version	V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#include "ipdefine.h"
#include "mcuip_drv.h"
#include "D_Para.h"

#include "ipdrv_siwdt.h"
#include "ipdrv_t32a.h"
#include "ipdrv_adc.h"
#include "ipdrv_enc.h"
#include "ipdrv_cg.h"
#include "ipdrv_uart.h"

#define DEFINE_APP
#include "initial.h"
#undef  DEFINE_APP


/*===================================================================*
	  Constant Data Definition
 *===================================================================*/
/*  */
#define	INT4KH_TMR_CH			TSB_T32A1
#define	INT4KH_IRQ_NUMBER		INTT32A01AC_IRQn
#define INT4KH_IRQ_LEVEL		INT4KH_LEVEL
#define	TMR_DIV8_PRD(x)			(uint16_t)( (uint32_t)x * cIMCLK_FRQ / 8)
#define	TMR_PRD_4KH				TMR_DIV8_PRD(250)

/**
 *********************************************************************************************
 * @brief		Initialize 4kH interval Timer.
 *
 * @return		none
 *********************************************************************************************
 */
void init_Timer_interval4kH(void)
{
	T32A_InitTypeDef t32a_00_a = {0};

	/* enable clock supply to T32A00 */
	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_T32A1, ENABLE);

	t32a_00_a.CounterMode = T32A_CNT_MOD_UP;
	t32a_00_a.ClkSrc = T32A_CLK_SRC_PRESCALER;
	t32a_00_a.ClkDiv = T32A_CLK_DIV_8;
	t32a_00_a.Compare1Value = TMR_PRD_4KH;		/* periodic time is 4kHz set to RGA1 */
	t32a_00_a.ReloadValue = 0x0000U;			/* reload register is 0x0000 */

	/* Set 16 bit mode for timer A */
	T32A_SetMode(INT4KH_TMR_CH, MODE_16);
	T32A_Enable(INT4KH_TMR_CH, TIMER_A);
	/* initialize timer A channel 0 */
	T32A_Init(INT4KH_TMR_CH, TIMER_A, &t32a_00_a);

	/* Set reload condition when counter match with RGA1 */
	T32A_SetReloadCond(INT4KH_TMR_CH, TIMER_A, T32A_COND_MATCH_COMPARE1);
	/* compare0 timing is not used in T32A_INTERVAL_TIMER mode */
	T32A_SetINTMask(INT4KH_TMR_CH, TIMER_A, T32A_INTMASK_MATCH_COMPARE0);
	/* Enable interrupt T32A Timer A channel 0 */
	NVIC_SetPriority(INT4KH_IRQ_NUMBER, INT4KH_IRQ_LEVEL);
	NVIC_ClearPendingIRQ(INT4KH_IRQ_NUMBER);
	NVIC_EnableIRQ(INT4KH_IRQ_NUMBER);

	/* counter start */
	T32A_SetRunStateBySW(INT4KH_TMR_CH, TIMER_A, T32A_START_BY_SW);
}

/**
 *********************************************************************************************
 * @brief	   Initialize encoder
 *
 * @return	  none
 *********************************************************************************************
 */
void init_ENCen(void)
{

	/******** Setting for FAN *********************************************/
#if defined(__USE_ENCODER_FAN)

	ENC_InitTypeDef m_enc;

	CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_A_ENC, ENABLE);

	m_enc.ModeType = ENC_ENCODER_MODE;
	m_enc.PhaseType = ENC_TWO_PHASE;
	m_enc.EdgeType = ENC_RISING_EDGE;
	m_enc.ZphaseStatus = ENC_ZPHASE_ENABLE;
	m_enc.SetClearCount = ENC_CLEAR_INT;
	m_enc.UpDownType = ENC_COUNT_UP;
	m_enc.CountOperation = ENC_COUNT_CONTINUES;
	m_enc.CompareMode = ENC_COMPARE_MATCH;
	m_enc.DetectionDirection = ENC_CW_CCW_EDGE;
	m_enc.SkipDetection = ENC_DETECTION_DISABLE;
	m_enc.SelectCapture = ENC_CAPTURE_CLEAR;
	m_enc.PulseDivFactor = ENC_PULSE_DIV1;

	ENC_Init(TSB_EN0, &m_enc);
	ENC_SetCounterReload(TSB_EN0, cENC_PULSE_NUM_FAN * cENCMULTIPLE - 1);
	ENC_SetSampleClk(TSB_EN0, ENC_SAMPLE_CLK_FSYS_8);
	ENC_SetNoiseCancelTime(TSB_EN0, (uint32_t)(cENC_NOISE_TIME_FAN * (cIMCLK_FRQ >> ENC_SAMPLE_CLK_FSYS_8)));

	CG_SetFsysSupplyPeriphA(CG_FSYS_PERIPH_PORTG, ENABLE);
	TSB_PG->FR5 |= BIT8(0, 0, 0, 0, 0, 1, 1, 1);	/* PG0,PG1,PG2 as ENC0A,ENC0B,ENC0Z */
	TSB_PG->IE |= BIT8(0, 0, 0, 0, 0, 1, 1, 1); /* PG0,PG1,PG2 as input,input,input */

	/* Enable ENC */
	ENC_Enable(TSB_EN0);

#endif			  /* __USE_ENCODER_FAN */

}

/**
 *********************************************************************************************
 * @brief		Initialize ADC
 *
 * @return		none
 *********************************************************************************************
 */
void init_ADCen(void)
{
	AD_InitTypeDef adini;

/******** Setting for COMP *********************************************/
#if defined(__CONTROL_MOTOR_COMP)
	adini.shunt = cSHUNT_TYPE_COMP;
	adini.iuch = cADCH_CURRENT_U_COMP;
	adini.ivch = cADCH_CURRENT_V_COMP;
	adini.iwch = cADCH_CURRENT_W_COMP;
	adini.idcch = cADCH_CURRENT_IDC_COMP;
	adini.vdcch = cADCH_VDC_COMP;
	adini.pmd_ch = cPMD1;
	adini.pints = cPINTS_B;
	adini.p_adreg0 = &Motor_comp.drv.ADxREG0;
	adini.p_adreg1 = &Motor_comp.drv.ADxREG1;
	adini.p_adreg2 = &Motor_comp.drv.ADxREG2;
	adini.p_adreg3 = &Motor_comp.drv.ADxREG3;
#if (defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
	IP_ADC_init(TSB_ADB, &adini);
#else
	IP_ADC_init(TSB_ADA, &adini);
#endif
	NVIC_SetPriority(INTADBPDB_IRQn, INT_ADC_LEVEL);
	NVIC_ClearPendingIRQ(INTADBPDB_IRQn);
	NVIC_EnableIRQ(INTADBPDB_IRQn);

#endif /* __CONTROL_MOTOR_COMP */


/******** Setting for FAN *********************************************/
#if defined(__CONTROL_MOTOR_FAN)
	adini.shunt = cSHUNT_TYPE_FAN;
	adini.iuch = cADCH_CURRENT_U_FAN;
	adini.ivch = cADCH_CURRENT_V_FAN;
	adini.iwch = cADCH_CURRENT_W_FAN;
	adini.idcch = cADCH_CURRENT_IDC_FAN;
	adini.vdcch = cADCH_VDC_FAN;
	adini.pmd_ch = cPMD0;
	adini.pints = cPINTS_A;
	adini.p_adreg0 = &Motor_fan.drv.ADxREG0;
	adini.p_adreg1 = &Motor_fan.drv.ADxREG1;
	adini.p_adreg2 = &Motor_fan.drv.ADxREG2;
	adini.p_adreg3 = &Motor_fan.drv.ADxREG3;
	IP_ADC_init(TSB_ADA, &adini);

#endif /* __CONTROL_MOTOR_FAN */

/******** Setting for ADC Unit *********************************************/
#if defined(__ADCZ)

	/* Set PLL for ADCLK */
	/* IP_PLL_ADCLK_init() */ /* Set by startup. So, this code is commented out here. */

#if (defined(__TMPM4KNA_H__) || defined(__TMPM4KLA_H__))
	/* Set ADC Unit A,B,C Clock */
	ADC_SetClk(TSB_ADA, ADC_SAMPLING_PERIOD_2, ADC_SCLK_1);
	ADC_SetClk(TSB_ADB, ADC_SAMPLING_PERIOD_2, ADC_SCLK_1);
	ADC_SetClk(TSB_ADC, ADC_SAMPLING_PERIOD_2, ADC_SCLK_1);
	/* Enable ADC Unit A,B,C */
	ADC_Enable(TSB_ADA, ADC_CLKMOD_160M, ADC_REFSEL_AIN, ADC_RCUT_NORMAL);
	ADC_Enable(TSB_ADB, ADC_CLKMOD_160M, ADC_REFSEL_AIN, ADC_RCUT_NORMAL);
	ADC_Enable(TSB_ADC, ADC_CLKMOD_160M, ADC_REFSEL_AIN, ADC_RCUT_NORMAL);
#endif
#endif /* __ADCZ */


/******** Setting for AMP Unit *********************************************/
	IP_AMP_init();

}

/**
 *********************************************************************************************
 * @brief		Initialize PMD
 *
 * @return		none
 *********************************************************************************************
 */
void init_PMDen(void)
{
	PMD_InitTypeDef	pmdini;

/******** Setting for COMP *********************************************/
#if defined(__CONTROL_MOTOR_COMP)
	pmdini.shunt = cSHUNT_TYPE_COMP;
	pmdini.poll = cPOLL_COMP;
	pmdini.polh = cPOLH_COMP;
	pmdini.pwmrate = PWMRate(cPWMPRD_COMP);
	pmdini.deadtime = DeadTime(cDEADTIME_COMP);
#if defined(__USE_VE_COMP)
	pmdini.busmode = WBUF_VE;
#else
	pmdini.busmode = WBUF_BUS;
#endif

	IP_PMD_init(TSB_PMD1, &pmdini);
	NVIC_SetPriority(INTEMG1_IRQn, INT_EMG1_LEVEL); /* EMG level set */
	NVIC_ClearPendingIRQ(INTEMG1_IRQn);
	NVIC_EnableIRQ(INTEMG1_IRQn);   /* INTEMGx enable */
#endif /* __CONTROL_MOTOR_COMP */


/******** Setting for FAN *********************************************/
#if defined(__CONTROL_MOTOR_FAN)
	pmdini.shunt = cSHUNT_TYPE_FAN;
	pmdini.poll = cPOLL_FAN;
	pmdini.polh = cPOLH_FAN;
	pmdini.pwmrate = PWMRate(cPWMPRD_FAN);
	pmdini.deadtime = DeadTime(cDEADTIME_FAN);
#if defined(__USE_VE_FAN)
	pmdini.busmode = WBUF_VE;
#else
	pmdini.busmode = WBUF_BUS;
#endif

	IP_PMD_init(TSB_PMD0, &pmdini);
	NVIC_SetPriority(INTEMG0_IRQn, INT_EMG0_LEVEL); /* EMG level set */
	NVIC_ClearPendingIRQ(INTEMG0_IRQn);
	NVIC_EnableIRQ(INTEMG0_IRQn);   /* INTEMGx enable */
#endif /* __CONTROL_MOTOR_FAN */

}


/**
 *********************************************************************************************
 * @brief	   Initialize VE.
 *
 * @return	  none
 *********************************************************************************************
 */
void init_VEen(void)
{

/******** Setting for FAN *********************************************/
#if defined(__CONTROL_MOTOR_FAN) && defined(__USE_VE_FAN)

	VE_InitTypeDef	veini;

	veini.shunt = cSHUNT_TYPE_FAN;
	veini.reptime = cREPTIME_FAN;
	veini.trgmode = cTRGMODE0_INI;
	veini.tpwm = (uint16_t)(cPWMPRD_FAN * 1e-6 * cHZ_MAX_FAN * (1 << 16U));

	veini.idkp = (q15_t)(FIX_15 * cID_KP_FAN * cA_MAX_FAN / cV_MAX_FAN);
	veini.idki = (q15_t)(FIX_15 * cID_KI_FAN * cCTRL_PRD_FAN * cA_MAX_FAN / cV_MAX_FAN);
	veini.idkg = cCIDKG;
	veini.iqkp = (q15_t)(FIX_15 * cIQ_KP_FAN * cA_MAX_FAN / cV_MAX_FAN);
	veini.iqki = (q15_t)(FIX_15 * cIQ_KI_FAN * cCTRL_PRD_FAN * cA_MAX_FAN / cV_MAX_FAN);
	veini.iqkg = cCIQKG;

	veini.zero_offset = ADC_V2HEX(cSHUNT_ZERO_OFFSET_FAN);

	NVIC_SetPriority(INTVCN0_IRQn, INT_VC_LEVEL);		/* Interruption level set */
	NVIC_ClearPendingIRQ(INTVCN0_IRQn);
	NVIC_EnableIRQ(INTVCN0_IRQn);						/* INTVCNx enable */

	IP_VE_init(TSB_VE0, &veini);

#endif /* defined(__CONTROL_MOTOR_FAN) && defined(__USE_VE_FAN) */

}

/**
 *********************************************************************************************
 * @brief		Initialize WDT
 *
 * @return		none
 *********************************************************************************************
 */
void init_WDT(void)
{
	CG_SetFsysSupplyPeriphB(CG_FSYS_PERIPH_SIWDT, ENABLE);
	SIWDT_SetClockSource(SIWDT_CLOCK_SOURCE_FSYS_DIVIDE_FOUR);
	/* SIWDT_SetDetectTime(SIWDT_DETECT_TIME_EXP_29);*/	/* 25sec */
	/* SIWDT_SetDetectTime(SIWDT_DETECT_TIME_EXP_27);*/	/* 7sec */
	SIWDT_SetDetectTime(SIWDT_DETECT_TIME_EXP_25);	/* 2sec */
	SIWDT_SetClearWindow(SIWDT_CLEAR_WINDOW_NOT_SPECIFIED);
	/* SIWDT_SetClearWindow(SIWDT_CLEAR_WINDOW_LATTER_A_HALF); */
	SIWDT_SetOperationAfterDetectMalFunction(SIWDT_OPERATION_INTERRUPT);
	SIWDT_Enable();
	SIWDT_SetProtectMode(SIWDT_NO_PROTECTION);
}


/**
 *********************************************************************************************
 * @brief	   Delay
 *
 * @return	  none
 *********************************************************************************************
 */
void Delay(uint16_t _cnt)
{
	uint16_t m, n;
	for (m = 0; m < _cnt; m++)
	{
		for (n = 0; n < 2001; n++);		/* 100us delay */
	}
}

/*********************************** END OF FILE ******************************/
