/**
 ****************************************************************************
 * @file	D_Driver.c
 * @brief   Motor vector control driver Source File
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#include <stdlib.h>
#include "ipdefine.h"
#include "mcuip_drv.h"
#include "D_Para.h"
#include "E_Sub.h"
#include "HPFC_drv.h"
#include "HPFC_para.h"
#include "ipdrv_adc.h"
#include "D_Driver.h"

#define DEFINE_APP
#include "D_Driver.h"
#undef	DEFINE_APP

/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
/*===================================================================*
	  Local Proto Type Definition
 *===================================================================*/
/**
 *********************************************************************************************
 * @brief		Detect _motor position
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void D_Detect_Rotor_Position(vector_t* const _motor)
{

	q31_t temp32_r, temp32_Lq;
	q31_t temp32_OmegaErrI, temp32_OmegaErrP;
	q31_t omega_calc;

	/**
	 * @note	Voltage equation
	 * 			Vd = R*Id + Ld*pId - OMEGA-est*Lq + Ed
	 * 			  (p = d/dt, Id=Constant value -> pId = 0)
	 */
	if (_motor->drv.vector_cmd.F_vcomm_Edetect == SET)
	{
		/* Voltage drop of resistance	 V = R * I */
		temp32_r = (_motor->para.motor.r * (q31_t)_motor->drv.Id) << 1;


		/* Voltage drop of inductance	 V = 2PIfL * I */
		temp32_Lq = (((_motor->drv.omega.half[1] * _motor->para.motor.Lq))) >> 15;		/* Q15 * __Q_FORMAT_MOTOR_L -> Q15 */
		temp32_Lq = (temp32_Lq * _motor->drv.Iq) << 1;								/* Q15 * Q15 -> Q31 */

		/* d-axis induced voltage */
		/* Ed = Vd - R*Id + OMEGA-est*Lq */
		_motor->drv.Ed = (q15_t)((_motor->drv.Vd.word - temp32_r + temp32_Lq + cROUND_BIT15) >> 16);

		/**
		 * @note	OMEGAest = OMEGAref - K*Ed
		 */
		/* Set integration value */
#if defined(__POSITION_KI_RANGE)
		/* OmegaErr(I) * Hz_MAX = FIX_12 * Position_Ki * CtrlPrd * Ed * V_MAX */
		temp32_OmegaErrI = (_motor->para.pos.ki * (q31_t)_motor->drv.Ed) << 4;
#else
		/* OmegaErr(I) * Hz_MAX = FIX_15 * Position_Ki * CtrlPrd * Ed * V_MAX */
		temp32_OmegaErrI = (_motor->para.pos.ki * (q31_t)_motor->drv.Ed) << 1;
#endif

		if (_motor->drv.omega_com.half[1] >= 0)
		{
			_motor->drv.Ed_I = __QSUB(_motor->drv.Ed_I, temp32_OmegaErrI);
		}
		else
		{
			_motor->drv.Ed_I = __QADD(_motor->drv.Ed_I, temp32_OmegaErrI);
		}

		/* Set proportionality value */
#if defined(__POSITION_KP_RANGE)
		/* OmegaErr(P) * Hz_MAX = FIX_12 * Position_Kp * Ed * V_MAX */
		temp32_OmegaErrP = (_motor->para.pos.kp * (q31_t)_motor->drv.Ed) << 4;
#else
		/* OmegaErr(P) * Hz_MAX = FIX_15 * Position_Kp * Ed * V_MAX */
		temp32_OmegaErrP = (_motor->para.pos.kp * (q31_t)_motor->drv.Ed) << 1;
#endif

		if (_motor->drv.omega_com.half[1] >= 0)
		{
			_motor->drv.Ed_PI = __QSUB(_motor->drv.Ed_I, temp32_OmegaErrP);
		}
		else
		{
			_motor->drv.Ed_PI = __QADD(_motor->drv.Ed_I, temp32_OmegaErrP);
		}

		/* Speed calculation value = Speed command value + Error PI value */
		omega_calc = __QADD(_motor->drv.omega_com.word, _motor->drv.Ed_PI);

		if (_motor->drv.vector_cmd.F_vcomm_omega == SET)
		{
			if (_motor->drv.command.encoder == CLEAR)
			{
				_motor->drv.omega.word = omega_calc;
			}
		}
		else
		{
			/* Initialize of proportionality value, integration value and Omega */
			_motor->drv.omega.word = _motor->drv.omega_com.word;
			_motor->drv.Ed_I = 0;
			_motor->drv.Ed_PI = 0;
		}

		/* Position estimation */
		if (_motor->drv.vector_cmd.F_vcomm_theta == SET)
		{
			if (_motor->drv.command.encoder == CLEAR)
			{
				/* estimated THETAn = THETAn-1 + OMEGA * t */
				_motor->drv.theta.word += ((_motor->drv.omega.half[1] * _motor->para.pos.ctrlprd) << 1);
			}
		}
		else
		{
			_motor->drv.theta.word = (q31_t)(_motor->drv.theta_com << 16);
		}
	}
	else
	{
		_motor->drv.theta.word = (q31_t)(_motor->drv.theta_com << 16);
		_motor->drv.Ed = 0;
	}
}

/**
 *********************************************************************************************
 * @brief		Speed control ( Caricurate q-axis current )
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void D_Control_Speed(vector_t* const _motor)
{
	/* Determination of R_Iq_ref */
	if (_motor->drv.vector_cmd.F_vcomm_current == SET)
	{
		q31_t temp32_PI;
		q31_t temp_iqref;

		/* Speed deviation */
		_motor->drv.omega_dev = (q15_t)(__QSUB(_motor->drv.omega_com.word, _motor->drv.omega.word) >> 16);

		/* Set proportionality value */
#if defined(__SPEED_KI_RANGE)
		temp32_PI = ((q31_t)_motor->para.spd.ki * _motor->drv.omega_dev) << 4;
#else
		temp32_PI = ((q31_t)_motor->para.spd.ki * _motor->drv.omega_dev) << 1;
#endif
		_motor->drv.Iq_ref_I.word = __QADD(_motor->drv.Iq_ref_I.word, temp32_PI);

		/* Limit value excess */
		if (_motor->drv.Iq_ref_I.word > _motor->para.iq_lim)
		{
			_motor->drv.Iq_ref_I.word = _motor->para.iq_lim;
		}
		else if (_motor->drv.Iq_ref_I.word < -_motor->para.iq_lim)
		{
			_motor->drv.Iq_ref_I.word = -_motor->para.iq_lim;
		}


		/* Set integration value */
#if defined(__SPEED_KP_RANGE)
		temp32_PI = ((q31_t)_motor->para.spd.kp * _motor->drv.omega_dev) << 4;
#else
		temp32_PI = ((q31_t)_motor->para.spd.kp * _motor->drv.omega_dev) << 1;
#endif
		temp_iqref = __QADD(_motor->drv.Iq_ref_I.word, temp32_PI);
		/* Limit value excess */
		if (temp_iqref > _motor->para.iq_lim)
		{
			_motor->drv.Iq_ref = (q15_t)(_motor->para.iq_lim >> 16);
		}
		else if (temp_iqref < -_motor->para.iq_lim)
		{
			_motor->drv.Iq_ref = (q15_t)(-_motor->para.iq_lim >> 16);
		}
		else
		{
			_motor->drv.Iq_ref = temp_iqref >> 16;
		}
	}
	else
	{
		/* Without Speed control. R_Iq_com -> R_Iq_ref */
		if (abs(_motor->drv.Iq_com.word) >= _motor->para.iq_lim)
		{
			/* Limit value excess */
			if (_motor->drv.Iq_com.word >= 0)
			{
				_motor->drv.Iq_ref = (q15_t)(_motor->para.iq_lim >> 16);
				_motor->drv.Iq_ref_I.word = _motor->para.iq_lim;
			}
			else
			{
				_motor->drv.Iq_ref = (q15_t)(-_motor->para.iq_lim >> 16);
				_motor->drv.Iq_ref_I.word = -_motor->para.iq_lim;
			}
		}
		else
		{
			/* Initialize of proportionality value and integration value */
			_motor->drv.Iq_ref = _motor->drv.Iq_com.half[1];
			_motor->drv.Iq_ref_I.word = _motor->drv.Iq_com.word;
		}

	}

	/* Determination of R_Id_ref */
	if (abs(_motor->drv.Id_com.word) >= _motor->para.id_lim)
	{
		/* Limit value excess */
		if (_motor->drv.Id_com.word >= 0)
		{
			_motor->drv.Id_ref = (q15_t)(_motor->para.id_lim >> 16);
		}
		else
		{
			_motor->drv.Id_ref = (q15_t)(-_motor->para.id_lim >> 16);
		}
	}
	else
	{
		_motor->drv.Id_ref = _motor->drv.Id_com.half[1];
	}

}

/**
 ********************************************************************************************
 * @brief		D_SetZeroCurrentByVE.
 *
 * @param[in]	vector_t* const	_motor	.
 * @param[in]	const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		none
 ********************************************************************************************
 */
void D_SetZeroCurrentByVE(vector_t* const _motor, TSB_VE_TypeDef * pVE)
{
	if (_motor->stage.main == cStop)
	{
		if (_motor->shunt_type == c3shunt)
		{
			uint32_t	adc_ia, adc_ib, adc_ic;
			VE_GetCurrentAdcData(pVE, &adc_ia, &adc_ib, &adc_ic);
			_motor->drv.Iao_ave.word += (((int32_t)(adc_ia << 16) - _motor->drv.Iao_ave.word) >> cIXO_AVE);
			_motor->drv.Ibo_ave.word += (((int32_t)(adc_ib << 16) - _motor->drv.Ibo_ave.word) >> cIXO_AVE);
			_motor->drv.Ico_ave.word += (((int32_t)(adc_ic << 16) - _motor->drv.Ico_ave.word) >> cIXO_AVE);
			VE_SetZeroCurrentData(pVE, _motor->drv.Iao_ave.half[1], _motor->drv.Ibo_ave.half[1], _motor->drv.Ico_ave.half[1]);
		}
		else if (_motor->shunt_type == c1shunt)
		{
			ADC_Result	i_adc = {0};
			uint32_t	adc_ia;

			i_adc = ADC_GetConvertResult(TSB_ADA, ADC_REG0);
			i_adc.All &= 0xfff0;

			_motor->drv.Iao_ave.word += (((int32_t)(i_adc.All << 16) - _motor->drv.Iao_ave.word) >> cIXO_AVE);

			adc_ia = _motor->drv.Iao_ave.half[1];
			VE_SetZeroCurrentData(pVE, adc_ia, adc_ia, adc_ia);
		}
	}
}

/**
 *********************************************************************************************
 * @brief		D_SetZeroCurrent
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void D_SetZeroCurrent(vector_t* const _motor)
{
	pfc_t *const _pfc = &Hpfc;

	if (_motor->stage.main == cStop)
	{
		if (_motor->shunt_type == c3shunt)
		{
			uint32_t	adc_ia, adc_ib, adc_ic;

			__disable_irq();
			adc_ia = _motor->drv.Ia_adc;
			adc_ib = _motor->drv.Ib_adc;
			adc_ic = _motor->drv.Ic_adc;
			__enable_irq();

			_motor->drv.Iao_ave.word += (((int32_t)(adc_ia << 16) - _motor->drv.Iao_ave.word) >> cIXO_AVE);
			_motor->drv.Ibo_ave.word += (((int32_t)(adc_ib << 16) - _motor->drv.Ibo_ave.word) >> cIXO_AVE);
			_motor->drv.Ico_ave.word += (((int32_t)(adc_ic << 16) - _motor->drv.Ico_ave.word) >> cIXO_AVE);
		}
		else if (_motor->shunt_type == c1shunt)
		{
			_motor->drv.Idco_ave.word += (((int32_t)(_motor->drv.Idc1_adc << 16) - _motor->drv.Idco_ave.word) >> cIXO_AVE);

			HPFC_CalZeroCurrent(_pfc);
		}
	}
}

/**
 *********************************************************************************************
 * @brief		Check over current.
 *
 * @param		q15_t			_ovc	current of over error value.
 * @param		q15_t			_iu		current of u phase.
 * @param		q15_t			_iv		current of v phase.
 * @param		q15_t			_iw		current of w phase.
 *
 * @return		bool	TRUE:Over current / FALSE:Nomal
 *********************************************************************************************
 */
bool D_Check_OverCurrent(q15_t _ovc, q15_t _iu, q15_t _iv, q15_t _iw)
{
	if ((abs(_iu) > _ovc)
		|| (abs(_iv) > _ovc)
		|| (abs(_iw) > _ovc))
	{
		return (true);
	}
	else
	{
		return (false);
	}
}

/**
 *********************************************************************************************
 * @brief		D_Check_DetectCurrentError
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none	
 *********************************************************************************************
 */
void D_Check_DetectCurrentError(vector_t* const _motor)
{
	uint32_t chkpls = _motor->drv.chkpls;
	uint8_t error = OFF;

	if (_motor->shunt_type == c1shunt)
	{
		int32_t max, med, min;
		int32_t dif1, dif2;

		min = _motor->drv.DutyU;
		med = _motor->drv.DutyV;
		max = _motor->drv.DutyW;
		E_Sort3(&min, &med, &max);

		dif1 = (max - med) / 2;
		dif2 = (med - min) / 2;

		if (_motor->drv.command.modul == 0)
		{
			/* 3 phase modulation */
			/* Check difference of duty width */
			if ((dif1 < chkpls) || (dif2 < chkpls))
			{
				error = ON;
			}
		}
		else
		{
			/* 2 phase modulation */
			/* Check duty width */
			if (med < chkpls)
			{
				error = ON;
			}
			/* Check difference of duty width */
			if (dif1 < chkpls)
			{
				error = ON;
			}
		}
	}

	if (error == ON)
	{
		_motor->drv.idetect_error = 1;
	}
	else
	{
		_motor->drv.idetect_error = 0;
	}
}

/**
 *********************************************************************************************
 * @brief		D_GetMotorCurrentPowerVolt
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none	
 *********************************************************************************************
 */
void D_GetMotorCurrentPowerVolt(vector_t* const _motor)
{
	/* Get Power Supply Voltage */
	if ((_motor->para.TrgPosMd == cTRGPOS_1SHUNT_SECOND_HALF)  || (_motor->para.TrgPosMd == cTRGPOS_1SHUNT_FIRST_HALF))
	{

		if (_motor->para.TrgPosMd == cTRGPOS_1SHUNT_FIRST_HALF)
		{
			/* Acquire AD at the position in the first half of the PWM cycle. */
			AD_GetIdcFirstHalf(_motor);
		}
		else /* if (_motor->para.TrgPosMd == cTRGPOS_1SHUNT_SECOND_HALF) */
		{
			/* Acquire AD at the position in the second half of the PWM cycle. */
			AD_GetIdcSecondHalf(_motor);
		}

		switch(_motor->drv.Sector1)
		{
			case 0:
				_motor->drv.Ic_raw = (q15_t)(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc1_adc);
				_motor->drv.Ia_raw = (q15_t)(-(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc2_adc));
				_motor->drv.Ib_raw = (q15_t)(-_motor->drv.Ia_raw - _motor->drv.Ic_raw);
				break;
			case 1:
				_motor->drv.Ic_raw = (q15_t)(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc1_adc);
				_motor->drv.Ib_raw = (q15_t)(-(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc2_adc));
				_motor->drv.Ia_raw = (q15_t)(-_motor->drv.Ib_raw -_motor->drv.Ic_raw);
				break;
			case 2:
				_motor->drv.Ia_raw = (q15_t)(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc1_adc);
				_motor->drv.Ib_raw = (q15_t)(-(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc2_adc));
				_motor->drv.Ic_raw = (q15_t)(-_motor->drv.Ia_raw - _motor->drv.Ib_raw);
				break;
			case 3:
				_motor->drv.Ia_raw = (q15_t)(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc1_adc);
				_motor->drv.Ic_raw = (q15_t)(-(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc2_adc));
				_motor->drv.Ib_raw = (q15_t)(-_motor->drv.Ia_raw - _motor->drv.Ic_raw);
				break;
			case 4:
				_motor->drv.Ib_raw = (q15_t)(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc1_adc);
				_motor->drv.Ic_raw = (q15_t)(-(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc2_adc));
				_motor->drv.Ia_raw = (q15_t)(-_motor->drv.Ib_raw - _motor->drv.Ic_raw);
				break;
			case 5:
				_motor->drv.Ib_raw = (q15_t)(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc1_adc);
				_motor->drv.Ia_raw = (q15_t)(-(_motor->drv.Idco_ave.half[1] - _motor->drv.Idc2_adc));
				_motor->drv.Ic_raw = (q15_t)(-_motor->drv.Ia_raw - _motor->drv.Ib_raw);
				break;
			default:
				break;
		}
	}

	if ((_motor->stage.itr == ciStop) || (_motor->drv.idetect_error == 0))
	{
		_motor->drv.Ia = _motor->drv.Ia_raw;
		_motor->drv.Ib = _motor->drv.Ib_raw;
		_motor->drv.Ic = _motor->drv.Ic_raw;
	}
}

/**
 *********************************************************************************************
 * @brief		D_InputTransformation
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none	
 *********************************************************************************************
 */
void D_InputTransformation(vector_t* const _motor)
{
	if (_motor->drv.vector_cmd.F_vcomm_Idetect == SET)
	{
		/* When the current detection can not be performed normally, conversion is not performed and Id and Iq are not updated. */
		if (_motor->drv.idetect_error == 0)
		{
			/* abc/alpha_beta conversion (Clarke.t) */
			E_Clarke(_motor->drv.Ia, _motor->drv.Ib, _motor->drv.Ic, &_motor->drv.Ialpha, &_motor->drv.Ibeta);

			/* alpha_beta/dq conversion (Park.t) */
			E_Park(_motor->drv.Ialpha, _motor->drv.Ibeta, _motor->drv.theta.half[1], &_motor->drv.Id, &_motor->drv.Iq);
		}
	}
	else
	{
		_motor->drv.Id = 0;
		_motor->drv.Iq = 0;
	}
}

/**
 *********************************************************************************************
 * @brief		D_Control_Current
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none	
 *********************************************************************************************
 */
void D_Control_Current(vector_t* const _motor)
{
	if (_motor->drv.vector_cmd.F_vcomm_volt == SET)
	{
		if ((_motor->stage.itr == ciInitposition_v) || (_motor->stage.itr == ciForce_v))
		{
			_motor->drv.Vd.word = _motor->drv.Vd_I.word = _motor->drv.Vd_out;
			_motor->drv.Vq.word = _motor->drv.Vq_I.word = 0;
		}
		else
		{
			q31_t	temp_PI;

			/* Id deviation */
			_motor->drv.Id_dev = (q15_t)(_motor->drv.Id_ref - _motor->drv.Id);
#if defined(__CURRENT_DKI_RANGE)
			temp_PI = ((q31_t)_motor->para.crt.dki * _motor->drv.Id_dev) << 4;
#else
			temp_PI = ((q31_t)_motor->para.crt.dki * _motor->drv.Id_dev) << 1;
#endif
			_motor->drv.Vd_I.word = __QADD(_motor->drv.Vd_I.word, temp_PI);

#if defined(__CURRENT_DKP_RANGE)
			temp_PI = ((q31_t)_motor->para.crt.dkp * _motor->drv.Id_dev) << 4;
#else
			temp_PI = ((q31_t)_motor->para.crt.dkp * _motor->drv.Id_dev) << 1;
#endif
			_motor->drv.Vd.word = __QADD(_motor->drv.Vd_I.word, temp_PI);

			/* Iq deviation */
			_motor->drv.Iq_dev = (q15_t)(_motor->drv.Iq_ref - _motor->drv.Iq);
#if defined(__CURRENT_QKI_RANGE)
			temp_PI = ((q31_t)_motor->para.crt.qki * _motor->drv.Iq_dev) << 4;
#else
			temp_PI = ((q31_t)_motor->para.crt.qki * _motor->drv.Iq_dev) << 1;
#endif
			_motor->drv.Vq_I.word = __QADD(_motor->drv.Vq_I.word, temp_PI);

#if defined(__CURRENT_QKP_RANGE)
			temp_PI = ((q31_t)_motor->para.crt.qkp * _motor->drv.Iq_dev) << 4;
#else
			temp_PI = ((q31_t)_motor->para.crt.qkp * _motor->drv.Iq_dev) << 1;
#endif
			_motor->drv.Vq.word = __QADD(_motor->drv.Vq_I.word, temp_PI);
		}
	}
	else
	{
		/* Without current control. Vd_com,Vq_com -> Vd,Vq */
		_motor->drv.Vd.word = _motor->drv.Vd_I.word = _motor->drv.Vd_com.word;
		_motor->drv.Vq.word = _motor->drv.Vq_I.word = _motor->drv.Vq_com.word;
	}
}

/**
 *********************************************************************************************
 * @brief		D_OutputTransformation
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none	
 *********************************************************************************************
 */
void D_OutputTransformation(vector_t* const _motor)
{
	if (_motor->drv.vector_cmd.F_vcomm_onoff == CLEAR)
	{
		_motor->drv.Sector0 = _motor->drv.Sector1 = 0;
		_motor->drv.DutyU  = _motor->drv.DutyV   = _motor->drv.DutyW = 0;
	}
	else
	{
		/*
			It converts from dq-axis voltage (Vd, Vq) to alpha_beta-axis voltage (Valpha, Vbeta) by reverse park conversion.
		*/
		E_InvPark(_motor->drv.Vd.word, _motor->drv.Vq.word, _motor->drv.theta.half[1], &_motor->drv.Valpha.word, &_motor->drv.Vbeta.word);

		/*
			Calculate sectors from alpha_beta axis voltage (Valpha, Vbeta).
			Sectors before calculation are saved in the last sector value and used for motor current acquisition.
		*/
		_motor->drv.Sector1 = _motor->drv.Sector0;
		_motor->drv.Sector0 = D_CalSector(_motor->drv.Valpha.word, _motor->drv.Vbeta.word);

		/*
			Convert from alpha_beta axis voltage (Valpha, Vbeta) to 3-phase duty (DutyV, DutyW, DutyW) by space vector modulation or reverse clark conversion.
		*/
		if (_motor->para.PhCvMd == cPHCVMD_SVM)
		{
			D_SVM(_motor->drv.Valpha.word, _motor->drv.Vbeta.word, _motor->drv.Vdc, _motor->drv.Sector0, _motor->drv.command.modul,
				  (uint16_t *)&_motor->drv.DutyU, (uint16_t *)&_motor->drv.DutyV, (uint16_t *)&_motor->drv.DutyW);
		}
		else if (_motor->para.PhCvMd == cPHCVMD_I_CLARKE)
		{
			/* InvClarke() not supported */
		}
	}
}

/**
 ********************************************************************************************
 * @brief		To determine the output mode.
 *
 * @param[in]	vector_t* const	_motor	.
 * @param[in]	const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		none
 ********************************************************************************************
 */
void D_DecisionOutputMode(vector_t* const _motor, TSB_PMD_TypeDef* const PMDx)
{
	uint32_t tmp_mdout;

	if (_motor->drv.vector_cmd.F_vcomm_onoff == CLEAR)
	{
		tmp_mdout = cMDOUT_OFF;
	}
	else
	{
		if (_motor->drv.state.all == 0)						/* No error? */
		{
			if (_motor->stage.itr == ciBootstrap)
			{
				tmp_mdout = cMDOUT_BOOTSTRAP;
			}
			else
			{
				tmp_mdout = cMDOUT_ON;
			}
		}
		else
		{
			tmp_mdout = cMDOUT_OFF;
		}
	}

	PMD_SetOutputMode(PMDx, tmp_mdout);

}

/**
 *********************************************************************************************
 * @brief		D_CalSector
 *
 * @param		q31_t 		_valpha		alpha-axis Valpha
 * @param		q31_t 		_vbeta		beta-axis Vbeta
 *
 * @return		sector	
 *********************************************************************************************
 */
uint8_t D_CalSector(q31_t _valpha, q31_t _vbeta)
{
	#define	cCAL_1_DIV_SQRT3	(int32_t)(FIX_15 * 1 / 1.73205f)	/* Q15 1/sqrt3 */

	uint8_t sector = 0;
	q31_t temp_vbeta = ((((int64_t)abs(_vbeta)) * cCAL_1_DIV_SQRT3) >> 15);

	if ((_valpha >= 0) && (_vbeta >= 0))
	{
		sector = (_valpha >= temp_vbeta)? 0: 1;
	}
	else if ((_valpha < 0) && (_vbeta >= 0))
	{
		sector = (abs(_valpha) < temp_vbeta)? 1: 2;
	}
	else if ((_valpha < 0) && (_vbeta < 0))
	{
		sector = (abs(_valpha) < temp_vbeta)? 4: 3;
	}
	else if ((_valpha >= 0) && (_vbeta < 0))
	{
		sector = (_valpha < temp_vbeta)? 4: 5;
	}
	return(sector);
}

/**
 *********************************************************************************************
 * @brief		D_SVM
 *
 * @param		q31_t 		_valpha		alpha-axis Valpha
 * @param		q31_t 		_vbeta		beta-axis Vbeta
 * @param		q15_t	 	_vdc		DC voltage
 * @param		uint8_t 	_sector		sector
 * @param		uint8_t 	_modul		modul
 * @param		uint16_t* 	_p_vu		U phase duty ratio storage address
 * @param		uint16_t* 	_p_vv		V phase duty ratio storage address
 * @param		uint16_t* 	_p_vw		W phase duty ratio storage address
 *
 * @return		none	
 *********************************************************************************************
 */
void D_SVM(q31_t _valpha, q31_t _vbeta, q15_t _vdc, uint8_t _sector, uint8_t _modul, uint16_t* _p_vu, uint16_t* _p_vv, uint16_t* _p_vw)
{
	#define cSVM_K				(int64_t)((1L<<15) * 3.0f / 2.0f)			/* Q15 3/2 */
	#define	cSVM_1_DIV_SQRT3	(int64_t)((1L<<15) * 1.0f / 1.7320508f)		/* Q15 1/sqrt3 */
	#define	cSVM_2_DIV_SQRT3	(int64_t)((1L<<15) * 2.0f / 1.7320508f)		/* Q15 2/sqrt3 */

	#define cSVM_100_PERCENT	(int32_t)(((1LL<<31) * 1) - 1) /* Q31 1.0 */

	int32_t temp_1, temp_2;
	q31_t d0, d1, d2;

	switch (_sector)
	{
		case 0:
			/* Sector0 */
			/* D1 = k * (Valpha - 1/sqrt3 * Vbeta) / Vdc */
			/* D2 = k * ( 2/sqrt3 * Vbeta) / Vdc */
			/* D0 = 100% - D1 - D2 */
			temp_1 = (int32_t)(__QSUB(_valpha, ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15)));
			temp_2 = (int32_t)((cSVM_2_DIV_SQRT3 * _vbeta) >> 15);
			d1 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_1 / _vdc)), 0, cSVM_100_PERCENT);
			d2 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_2 / _vdc)), 0, cSVM_100_PERCENT);
			d0 = (q31_t)Limit_sub(__QSUB(__QSUB(cSVM_100_PERCENT, d1), d2), 0, cSVM_100_PERCENT);
			/* Set Duty */
			if (_modul == 0)
			{
				/* 3phase modulation */
				/* U = D1 + D2 + (D0 / 2) */
				/* V = D2 + (D0 / 2) */
				/* W = D0 / 2 */
				*_p_vu = (uint16_t)(__QADD(__QADD((d0 / 2), d1), d2) >> 16);
				*_p_vv = (uint16_t)(__QADD((d0 / 2), d2) >> 16);
				*_p_vw = (uint16_t)((d0 / 2) >> 16);
			}
			else
			{
				/* 2phase modulation */
				/* U = D1 + D2 */
				/* V = D2 */
				/* W = 0 */
				*_p_vu = (uint16_t)(__QADD(d1, d2) >> 16);
				*_p_vv = (uint16_t)(d2 >> 16);
				*_p_vw = 0;
			}
			break;

		case 1:
			/* Sector1 */
			/* D1 = k * (-Valpha + 1/sqrt3 * Vbeta) / Vdc */
			/* D2 = k * (Valpha + 1/sqrt3 * Vbeta) / Vdc */
			/* D0 = 100% - D1 - D2 */
			temp_1 = (int32_t)__QADD((-_valpha), ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15));
			temp_2 = (int32_t)__QADD(_valpha, ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15));
			d1 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_1 / _vdc)), 0, cSVM_100_PERCENT);
			d2 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_2 / _vdc)), 0, cSVM_100_PERCENT);
			d0 = (q31_t)Limit_sub(__QSUB(__QSUB(cSVM_100_PERCENT, d1), d2), 0, cSVM_100_PERCENT);
			/* Set Duty */
			if (_modul == 0)
			{
				/* 3phase modulation */
				/* U = D2 + (D0 / 2) */
				/* V = D1 + D2 + (D0 / 2) */
				/* W = D0 / 2 */
				*_p_vu = (uint16_t)(__QADD((d0 / 2), d2) >> 16);
				*_p_vv = (uint16_t)(__QADD(__QADD((d0 / 2), d1), d2) >> 16);
				*_p_vw = (uint16_t)((d0 / 2) >> 16);
			}
			else
			{
				/* 2phase modulation */
				/* U = D2 */
				/* V = D1 + D2 */
				/* W = 0 */
				*_p_vu = (uint16_t)(d2 >> 16);
				*_p_vv = (uint16_t)(__QADD(d1, d2) >> 16);
				*_p_vw = 0;
			}
			break;

		case 2:
			/* Sector2 */
			/* D1 = k * ( 2/sqrt3 * Vbeta) / Vdc */
			/* D2 = k * (-Valpha - 1/sqrt3 * Vbeta) / Vdc */
			/* D0 = 100% - D1 - D2 */
			temp_1 = (int32_t)((cSVM_2_DIV_SQRT3 * _vbeta) >> 15);
			temp_2 = (int32_t)__QSUB((-_valpha), ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15));
			d1 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_1 / _vdc)), 0, cSVM_100_PERCENT);
			d2 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_2 / _vdc)), 0, cSVM_100_PERCENT);
			d0 = (q31_t)Limit_sub(__QSUB(__QSUB(cSVM_100_PERCENT, d1), d2), 0, cSVM_100_PERCENT);
			/* Set Duty */
			if (_modul == 0)
			{
				/* 3phase modulation */
				/* U = D0 / 2 */
				/* V = D1 + D2 + (D0 / 2) */
				/* W = D2 + (D0 / 2) */
				*_p_vu = (uint16_t)((d0 / 2) >> 16);
				*_p_vv = (uint16_t)(__QADD(__QADD((d0 / 2), d1), d2) >> 16);
				*_p_vw = (uint16_t)(__QADD((d0 / 2), d2) >> 16);
			}
			else
			{
				/* 2phase modulation */
				/* U = 0 */
				/* V = D1 + D2 */
				/* W = D2 */
				*_p_vu = 0;
				*_p_vv = (uint16_t)(__QADD(d1, d2) >> 16);
				*_p_vw = (uint16_t)(d2 >> 16);
			}
			break;

		case 3:
			/* Sector3 */
			/* D1 = k * (-2/sqrt3 * Vbeta) / Vdc */
			/* D2 = k * (-Valpha + 1/sqrt3 * Vbeta) / Vdc */
			/* D0 = 100% - D1 - D2 */
			temp_1 = (int32_t)((cSVM_2_DIV_SQRT3 * _vbeta) >> 15);
			temp_2 = (int32_t)__QADD((-_valpha), ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15));
			d1 = (q31_t)Limit_sub((int64_t)(cSVM_K * (-temp_1 /_vdc)), 0, cSVM_100_PERCENT);
			d2 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_2 / _vdc)), 0, cSVM_100_PERCENT);
			d0 = (q31_t)Limit_sub(__QSUB(__QSUB(cSVM_100_PERCENT, d1), d2), 0, cSVM_100_PERCENT);
			/* Set Duty */
			if (_modul == 0)
			{
				/* 3phase modulation */
				/* U = D0 / 2 */
				/* V = D2 + (D0 / 2) */
				/* W = D1 + D2 + (D0 / 2) */
				*_p_vu = (uint16_t)((d0 / 2) >> 16);
				*_p_vv = (uint16_t)(__QADD((d0 / 2), d2) >> 16);
				*_p_vw = (uint16_t)(__QADD(__QADD((d0 / 2), d1), d2) >> 16);
			}
			else
			{
				/* 2phase modulation */
				/* U = 0 */
				/* V = D2 */
				/* W = D1 + D2 */
				*_p_vu = 0;
				*_p_vv = (uint16_t)(d2 >> 16);
				*_p_vw = (uint16_t)(__QADD(d1, d2) >> 16);
			}
			break;

		case 4:
			/* Sector4 */
			/* D1 = k * (-Valpha - 1/sqrt3 * Vbeta) / Vdc */
			/* D2 = k * (Valpha - 1/sqrt3 * Vbeta) / Vdc */
			/* D0 = 100% - D1 - D2 */
			temp_1 = (int32_t)__QSUB((-_valpha), ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15));
			temp_2 = (int32_t)__QSUB(_valpha, ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15));
			d1 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_1 / _vdc)), 0, cSVM_100_PERCENT);
			d2 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_2 / _vdc)), 0, cSVM_100_PERCENT);
			d0 = (q31_t)Limit_sub(__QSUB(__QSUB(cSVM_100_PERCENT, d1), d2), 0, cSVM_100_PERCENT);
			/* Set Duty */
			if (_modul == 0)
			{
				/* 3phase modulation */
				/* U = D2 + (D0 / 2) */
				/* V = D0 / 2 */
				/* W = D1 + D2 + (D0 / 2) */
				*_p_vu = (uint16_t)(__QADD((d0 / 2), d2) >> 16);
				*_p_vv = (uint16_t)((d0 / 2) >> 16);
				*_p_vw = (uint16_t)(__QADD(__QADD((d0 / 2), d1), d2) >> 16);
			}
			else
			{
				/* 2phase modulation */
				/* U = D2 */
				/* V = 0 */
				/* W = D1 + D2 */
				*_p_vu = (uint16_t)(d2 >> 16);
				*_p_vv = 0;
				*_p_vw = (uint16_t)(__QADD(d1, d2) >> 16);
			}
			break;

		case 5:
			/* Sector5 */
			/* D1 = k * (Valpha + 1/sqrt3 * Vbeta) / Vdc */
			/* D2 = k * ( -2/sqrt3 Vbeta) / Vdc */
			/* D0 = 100% - D1 - D2 */
			temp_1 = (int32_t)__QADD(_valpha, ((cSVM_1_DIV_SQRT3 * _vbeta) >> 15));
			temp_2 = (int32_t)((cSVM_2_DIV_SQRT3 * _vbeta) >> 15);
			d1 = (q31_t)Limit_sub((int64_t)(cSVM_K * (temp_1 / _vdc)), 0, cSVM_100_PERCENT);
			d2 = (q31_t)Limit_sub((int64_t)(cSVM_K * (-temp_2 / _vdc)), 0, cSVM_100_PERCENT);
			d0 = (q31_t)Limit_sub(__QSUB(__QSUB(cSVM_100_PERCENT, d1), d2), 0, cSVM_100_PERCENT);
			/* Set Duty */
			if (_modul == 0)
			{
				/* 3phase modulation */
				/* U = D1 + D2 + (D0 / 2) */
				/* V = D0 / 2 */
				/* W = D2 + (D0 / 2) */
				*_p_vu = (uint16_t)(__QADD(__QADD((d0 / 2), d1), d2) >> 16);
				*_p_vv = (uint16_t)((d0 / 2) >> 16);
				*_p_vw = (uint16_t)(__QADD((d0 / 2), d2) >> 16);
			}
			else
			{
				/* 2phase modulation */
				/* U = D1 + D2 */
				/* V = 0 */
				/* W = D2 */
				*_p_vu = (uint16_t)(__QADD(d1, d2) >> 16);
				*_p_vv = 0;
				*_p_vw = (uint16_t)(d2 >> 16);
			}
			break;
		default:
			break;
	}
}

/**
 *********************************************************************************************
 * @brief		D_CalTrgTiming
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none	
 *********************************************************************************************
 */
void D_CalTrgTiming(TSB_PMD_TypeDef* const PMDx, vector_t* const _motor)
{
	if (_motor->drv.vector_cmd.F_vcomm_onoff == CLEAR)
	{
		_motor->drv.AdTrg0 = _motor->drv.AdTrg1  = 0;
	}
	else
	{
		uint32_t  dmin, dmid, dmax;

		switch(_motor->para.TrgPosMd)
		{
			case cTRGPOS_1SHUNT_SECOND_HALF:
				dmid = E_Med3(_motor->drv.DutyU, 
							  _motor->drv.DutyV, 
							  _motor->drv.DutyW);
				dmax = E_Max3(_motor->drv.DutyU, 
							  _motor->drv.DutyV, 
							  _motor->drv.DutyW);

				/* 2phase and 3phase modulation */
				 _motor->drv.AdTrg0 = dmid + _motor->para.TrgCompen;
				 _motor->drv.AdTrg1 = dmax + _motor->para.TrgCompen;
				break;

			case cTRGPOS_1SHUNT_FIRST_HALF:
				dmid = E_Med3(_motor->drv.DutyU, 
							  _motor->drv.DutyV, 
							  _motor->drv.DutyW);
				dmin = E_Min3(_motor->drv.DutyU, 
							  _motor->drv.DutyV, 
							  _motor->drv.DutyW);

				if (_motor->drv.command.modul == 0)
				{
					/* 3phase modulation */
					_motor->drv.AdTrg0 = (-dmid) + _motor->para.TrgCompen;
					_motor->drv.AdTrg1 = (-dmin) + _motor->para.TrgCompen;
				}
				else
				{
					/* 2phase modulation */
					_motor->drv.AdTrg0 = (-dmid) + _motor->para.TrgCompen;
					_motor->drv.AdTrg1 = dmid + _motor->para.TrgCompen;
				}
				break;

			case cTRGPOS_3SHUNT:
			{
				int32_t	trgtmp;
				
				if (_motor->para.TrgCompen > 0)
				{
					trgtmp = (1L<<15) + _motor->para.TrgCompen;
					if (trgtmp > (1L<<15))	trgtmp -= (1L<<16);
				}
				else
				{
					trgtmp = (1L<<15);
				}

				_motor->drv.AdTrg0 = trgtmp;
				_motor->drv.AdTrg1 = trgtmp;
			}
				break;

			default:
				break;
		}
	}
}

/**
 *********************************************************************************************
 * @brief		D_Power_Limit
 *
 * @param		pfc_t* const	_pfc			The structure address for pfc control.
 *
 * @return		none
 *********************************************************************************************
 */
void D_Power_Limit(pfc_t * const _pfc, vector_t* const _motor)
{

	_motor->power.Iac_temp += (_pfc->drv.Iac_raw - _motor->power.Iac_temp) >> 1;
	_motor->power.Vac_temp += (_pfc->drv.m_Vac - _motor->power.Vac_temp) >> 2;

	_motor->power.Iac_temp_sum += _motor->power.Iac_temp * _motor->power.Iac_temp;
	_motor->power.Vac_temp_sum += _motor->power.Vac_temp * _motor->power.Vac_temp;
	_motor->power.sum_cnt++;

	/* Power is caculated every 50HZ */
	if (_motor->power.start_power_flag == 1)
	{
		_motor->power.start_power_flag = 0;
		D_PowerCalculation(_motor);
		D_PowerCompensate(_motor);
	}

	/* Power limit simple decease speed */
	if (_motor->power.apparent_power > cPOWER_LIMIT)
	{
		if (++_motor->power.power_cnt > _motor->power.power_para)
		{
			_motor->power.power_cnt = 0;
			if (_motor->power.fdecrease == 0)
			{
				_motor->spd_power_limit = _motor->current_spd * cPOLEPAIR_COMP;
			}
			_motor->power.fdecrease = 1;
			_motor->power.cnt_t0++;
			if (_motor->power.cnt_t0 >= 3)
			{
				_motor->power.cnt_t0 = 4;
				_motor->power.cnt_t1++;
				if (_motor->power.cnt_t1 > cPOWER_DECREASE_SPD_FILTER)
				{
					_motor->spd_power_limit--;
					_motor->power.cnt_t1 = 0;
				}
			}
		}
	}
	else
	{
		_motor->power.power_cnt = 0;
		_motor->power.cnt_t0 = 0;
		_motor->power.cnt_t1 = 0;
		_motor->power.fdecrease = 0;
	}
	if (_motor->power.apparent_power > cPOWER_EMG)
	{
		if (++_motor->power.power_cnt1 > _motor->power.power_para)
		{
			_motor->power.power_cnt1 = 0;
			_motor->drv.state.flg.over_power = SET;
		}

	}
}

/**
 *********************************************************************************************
 * @brief		D_Power_Compensate
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void D_PowerCompensate(vector_t* const _motor)
{
	/* add power compensate 900W-1400(0-30W)/1400-2100(30-100W) */
	if (_motor->power.apparent_power > cPOWER_LINE1)
	{
		if (_motor->power.apparent_power < cPOWER_LINE2)
		{
			_motor->power.apparent_power_compensate = ((cPOWER_COMPENSATE_FACTOR1 - cPOWER_COMPENSATE_FACTOR0)
					* (_motor->power.apparent_power  - cPOWER_LINE1)) / (cPOWER_LINE2 - cPOWER_LINE1) + cPOWER_COMPENSATE_FACTOR0;
		}
		else if (_motor->power.apparent_power < cPOWER_LINE3)
		{
			_motor->power.apparent_power_compensate = (cCOMPENSATE_MAX - cPOWER_COMPENSATE_FACTOR1) * (_motor->power.apparent_power  - cPOWER_LINE2)
					/ (cPOWER_LINE3 - cPOWER_LINE2) + cPOWER_COMPENSATE_FACTOR1;
		}
		else
		{
			_motor->power.apparent_power_compensate = cCOMPENSATE_MAX;
		}
	}
	else
	{
		_motor->power.apparent_power_compensate = 0;
	}
	_motor->power.apparent_power =  _motor->power.apparent_power +  _motor->power.apparent_power_compensate;
}
/**
 *********************************************************************************************
 * @brief	   Eq cal
 *
 * @param	   q15_t   _id	 Id
 * @param	   q15_t   _iq	 Iq
 * @param	   q31_t   _vq	 Vq
 * @param	   q15_t   _omega  Omega
 *
 * @return	  q15_t   Eq
 *********************************************************************************************
 */
q15_t Cal_Eq(q15_t _id, q15_t _iq, q31_t _vq, q15_t _omega, uint8_t _type)
{
	q31_t temp32_r, temp32_L, temp32_id;
	q15_t eq;

	if (_type == cCOMP)
	{
		/* Voltage drop of resistance	V = R * I */
		temp32_r = ((q31_t)(FIX_15 * cMOTOR_R_COMP * cA_MAX_COMP / cV_MAX_COMP) * (q31_t)_iq) << 1;

		temp32_L = (q31_t)(FIX_15 * PAI2 * cA_MAX_COMP * cHZ_MAX_COMP * cMOTOR_LD_COMP / 1000 / cV_MAX_COMP);
		temp32_id = (((_omega * _id) << 1) + cROUND_BIT15) >> 16;
		temp32_L = (temp32_L * temp32_id) << 1;

		/* q-axis induced voltage */
		eq = (q15_t)((_vq - temp32_r - temp32_L + cROUND_BIT15) >> 16);
	}
	else
	{
		/* Voltage drop of resistance	V = R * I */
		temp32_r = ((q31_t)(FIX_15 * cMOTOR_R_FAN * cA_MAX_FAN / cV_MAX_FAN) * (q31_t)_iq) << 1;

		/* Voltage drop of inductance	V = 2paifL * I */
		temp32_L = (q31_t)(FIX_15 * PAI2 * cA_MAX_FAN * cHZ_MAX_FAN * cMOTOR_LD_FAN / 1000 / cV_MAX_FAN);
		temp32_id = (((_omega * _id) << 1) + cROUND_BIT15) >> 16;
		temp32_L = (temp32_L * temp32_id) << 1;

		/* q-axis induced voltage */
		eq = (q15_t)((_vq - temp32_r - temp32_L + cROUND_BIT15) >> 16);
	}

	return (eq);
}

/**
 *********************************************************************************************
 * @brief		D_PowerCalculation
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void  D_PowerCalculation(vector_t* const _motor)
{
	/* Iac eff */
	_motor->power.Iac_eff_temp =
		((uint16_t)sqrt(_motor->power.Iac_temp_sum / _motor->power.sum_cnt)) * _motor->power.pfc_max_current * 100 / (FIX_12 / 2);
	_motor->power.Iac_eff += (_motor->power.Iac_eff_temp - _motor->power.Iac_eff) >> _motor->Iac_eff_filter_shift3;

	/* Vac eff */
	_motor->power.Vac_eff_temp =
		((uint16_t)sqrt(_motor->power.Vac_temp_sum / _motor->power.sum_cnt)) * _motor->power.Vac_max / (FIX_12 / 2);
	_motor->power.Vac_eff += (_motor->power.Vac_eff_temp - _motor->power.Vac_eff) >> _motor->Vac_eff_filter_shift4;

	/* Apparent power */
	_motor->power.apparent_power_tmp = _motor->power.Iac_eff * _motor->power.Vac_eff / 1000;
	_motor->power.apparent_power +=
		(_motor->power.apparent_power_tmp - _motor->power.apparent_power);

	/* Clear sum */
	_motor->power.Iac_temp_sum = 0;
	_motor->power.Vac_temp_sum = 0;
	_motor->power.sum_cnt = 0;
}

/*********************************** END OF FILE ******************************/
