/**
 *******************************************************************************
 * @file    ipdrv_adc.c
 * @brief   This file provides API functions for M4Ky ADC driver.
 * @version V1.1
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 *
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "ipdrv_adc.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @defgroup ADC
  * @brief ADC driver modules
  * @{
  */

/** @defgroup ADC_Private_Defines
  * @{
  */
#define ADC_MASK_KEEP1BIT       ((uint32_t)0x00000001)

#define ADC_MASK_SINGLE         ((uint32_t)0x00000081)
#define ADC_MASK_CONTINUE       ((uint32_t)0x00000082)

/** @} */
/* End of group ADC_Private_Defines */

/** @defgroup ADC_Private_FunctionPrototypes
  * @{
  */

/** @} */
/* End of group ADC_Private_FunctionPrototypes */

/** @defgroup ADC_Private_Functions
  * @{
  */

static const volatile uint32_t *const ADA_REGx_Address[24U] = {
    &TSB_ADA->REG0, &TSB_ADA->REG1, &TSB_ADA->REG2, &TSB_ADA->REG3,
    &TSB_ADA->REG4, &TSB_ADA->REG5, &TSB_ADA->REG6, &TSB_ADA->REG7,
    &TSB_ADA->REG8, &TSB_ADA->REG9, &TSB_ADA->REG10, &TSB_ADA->REG11,
    &TSB_ADA->REG12, &TSB_ADA->REG13, &TSB_ADA->REG14, &TSB_ADA->REG15,
    &TSB_ADA->REG16, &TSB_ADA->REG17, &TSB_ADA->REG18, &TSB_ADA->REG19,
    &TSB_ADA->REG20, &TSB_ADA->REG21, &TSB_ADA->REG22, &TSB_ADA->REG23
};

static volatile uint32_t *const ADA_TSETx_Address[24U] = {
    &TSB_ADA->TSET0, &TSB_ADA->TSET1, &TSB_ADA->TSET2, &TSB_ADA->TSET3,
    &TSB_ADA->TSET4, &TSB_ADA->TSET5, &TSB_ADA->TSET6, &TSB_ADA->TSET7,
    &TSB_ADA->TSET8, &TSB_ADA->TSET9, &TSB_ADA->TSET10, &TSB_ADA->TSET11,
    &TSB_ADA->TSET12, &TSB_ADA->TSET13, &TSB_ADA->TSET14, &TSB_ADA->TSET15,
    &TSB_ADA->TSET16, &TSB_ADA->TSET17, &TSB_ADA->TSET18, &TSB_ADA->TSET19,
    &TSB_ADA->TSET20, &TSB_ADA->TSET21, &TSB_ADA->TSET22, &TSB_ADA->TSET23
};

static const volatile uint32_t *const ADB_REGx_Address[24U] = {
    &TSB_ADB->REG0, &TSB_ADB->REG1, &TSB_ADB->REG2, &TSB_ADB->REG3,
    &TSB_ADB->REG4, &TSB_ADB->REG5, &TSB_ADB->REG6, &TSB_ADB->REG7,
    &TSB_ADB->REG8, &TSB_ADB->REG9, &TSB_ADB->REG10, &TSB_ADB->REG11,
    &TSB_ADB->REG12, &TSB_ADB->REG13, &TSB_ADB->REG14, &TSB_ADB->REG15,
    &TSB_ADB->REG16, &TSB_ADB->REG17, &TSB_ADB->REG18, &TSB_ADB->REG19,
    &TSB_ADB->REG20, &TSB_ADB->REG21, &TSB_ADB->REG22, &TSB_ADB->REG23
};

static volatile uint32_t *const ADB_TSETx_Address[24U] = {
    &TSB_ADB->TSET0, &TSB_ADB->TSET1, &TSB_ADB->TSET2, &TSB_ADB->TSET3,
    &TSB_ADB->TSET4, &TSB_ADB->TSET5, &TSB_ADB->TSET6, &TSB_ADB->TSET7,
    &TSB_ADB->TSET8, &TSB_ADB->TSET9, &TSB_ADB->TSET10, &TSB_ADB->TSET11,
    &TSB_ADB->TSET12, &TSB_ADB->TSET13, &TSB_ADB->TSET14, &TSB_ADB->TSET15,
    &TSB_ADB->TSET16, &TSB_ADB->TSET17, &TSB_ADB->TSET18, &TSB_ADB->TSET19,
    &TSB_ADB->TSET20, &TSB_ADB->TSET21, &TSB_ADB->TSET22, &TSB_ADB->TSET23
};

static const volatile uint32_t *const ADC_REGx_Address[24U] = {
    &TSB_ADC->REG0, &TSB_ADC->REG1, &TSB_ADC->REG2, &TSB_ADC->REG3,
    &TSB_ADC->REG4, &TSB_ADC->REG5, &TSB_ADC->REG6, &TSB_ADC->REG7,
    &TSB_ADC->REG8, &TSB_ADC->REG9, &TSB_ADC->REG10, &TSB_ADC->REG11,
    &TSB_ADC->REG12, &TSB_ADC->REG13, &TSB_ADC->REG14, &TSB_ADC->REG15,
    &TSB_ADC->REG16, &TSB_ADC->REG17, &TSB_ADC->REG18, &TSB_ADC->REG19,
    &TSB_ADC->REG20, &TSB_ADC->REG21, &TSB_ADC->REG22, &TSB_ADC->REG23
};

static volatile uint32_t *const ADC_TSETx_Address[24U] = {
    &TSB_ADC->TSET0, &TSB_ADC->TSET1, &TSB_ADC->TSET2, &TSB_ADC->TSET3,
    &TSB_ADC->TSET4, &TSB_ADC->TSET5, &TSB_ADC->TSET6, &TSB_ADC->TSET7,
    &TSB_ADC->TSET8, &TSB_ADC->TSET9, &TSB_ADC->TSET10, &TSB_ADC->TSET11,
    &TSB_ADC->TSET12, &TSB_ADC->TSET13, &TSB_ADC->TSET14, &TSB_ADC->TSET15,
    &TSB_ADC->TSET16, &TSB_ADC->TSET17, &TSB_ADC->TSET18, &TSB_ADC->TSET19,
    &TSB_ADC->TSET20, &TSB_ADC->TSET21, &TSB_ADC->TSET22, &TSB_ADC->TSET23
};

/** @} */
/* End of group ADC_Private_Functions */


/** @defgroup ADC_Exported_Functions
  * @{
  */

/**
  * @brief  Set ADC prescaler output(SCLK) of the specified ADC unit.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *   TSB_ADA.
  * @param  Sampling_Period: Select ADC sample period time.
  *   This parameter can be one of the following values:
  *   ADC_SAMPLING_PERIOD_1, ADC_SAMPLING_PERIOD_2
  * @param  Prescaler_Output: Select ADC prescaler output.
  *   This parameter can be one of the following values:
  *   ADC_SCLK_1,   ADC_SCLK_2
  * @retval None
  * @register The used registers:
  *   ADCCLK<EXAZ[6:3]><VADCLK[2:0]>
  */
void ADC_SetClk(TSB_AD_TypeDef * ADx, uint32_t Sampling_Period, uint32_t Prescaler_Output)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));
    assert_param(IS_ADC_SAMPLING_PERIOD(Sampling_Period));
    assert_param(IS_ADC_PRESCALER(Prescaler_Output));

    tmp = Sampling_Period | Prescaler_Output;
    ADx->CLK = tmp;
}


/**
  * @brief  Enable the specified ADC unit.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *   TSB_ADA.
  * @param  ClkMod: Select clock mode.
  *   This parameter can be one of the following values:
  *   Conversion time. 1.0us:ADC_CLKMOD_160M, 1.0us:ADC_CLKMOD_120M, 1.1us:ADC_CLKMOD_80M
  * @param  Sel: Select VREFHx/AINn switching control.
  *   This parameter can be one of the following values:
  *   VREFHx:ADC_REFSEL_VREFH, AINn:ADC_REFSEL_AIN
  * @param  Rcut: Select Low power consumption mode.
  *   This parameter can be one of the following values:
  *   Normal:ADC_RCUT_NORMAL, Low power:ADC_RCUT_LOWPOWER
  * @retval None
  * @register The used registers:
  *   ADCMOD0<REFBSEL><RCUT><DACON>
  *   ADCCR0<ADEN>
  */
void ADC_Enable(TSB_AD_TypeDef * ADx, ADC_ClkMod ClkMod, ADC_RefSel Sel, ADC_RcutMode Rcut)
{
    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));
    assert_param(IS_ADC_CLKMOD(ClkMod));
    assert_param(IS_ADC_REFSEL(Sel));
    assert_param(IS_ADC_RCUTMODE(Rcut));

    ADx->MOD0 = (Rcut << 1U) | (1U << 0U);

    if (ClkMod ==ADC_CLKMOD_160M) {
        ADx->MOD1 = 0x00306122;
    } else if (ClkMod ==ADC_CLKMOD_120M) {
        ADx->MOD1 = 0x00308012;
    } else { 
		/* if (ClkMod ==ADC_CLKMOD_80M) */
        ADx->MOD1 = 0x00104011;
		/* other is not support. */
    }
	ADx->MOD2 = 0x00000000;

    if (ADx == TSB_ADA) {
	    TSB_ADA_CR0_ADEN = 1U;
	} else if (ADx == TSB_ADB) {
    	TSB_ADB_CR0_ADEN = 1U;
	} else {
    	TSB_ADC_CR0_ADEN = 1U;
	}
}

/**
  * @brief  Disable the specified ADC unit.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *   TSB_ADA, TSB_ADB.
  * @retval None
  * @register The used registers:
  *   ADCMOD0<REFBSEL><RCUT><DACON>
  *   ADCCR0<ADEN>
  */
void ADC_Disable(TSB_AD_TypeDef * ADx)
{
    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));

    if (ADx == TSB_ADA) {
	    TSB_ADA_CR0_ADEN = 0U;
	    TSB_ADA_MOD0_RCUT = 1U;
	    TSB_ADA_MOD0_DACON = 0U;
	} else if (ADx == TSB_ADB) {
	    TSB_ADB_CR0_ADEN = 0U;
	    TSB_ADB_MOD0_RCUT = 1U;
	    TSB_ADB_MOD0_DACON = 0U;
	} else {
	    TSB_ADC_CR0_ADEN = 0U;
	    TSB_ADC_MOD0_RCUT = 1U;
	    TSB_ADC_MOD0_DACON = 0U;
	}
}

/**
  * @brief  Start the specified ADC unit with software trigger or constant trigger.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *   TSB_ADA.
  * @param  Trg: Select trigger type.
  *   This parameter can be one of the following values:
  *   ADC_TRG_SINGLE, ADC_TRG_CONTINUE
  * @retval None
  * @register The used registers:
  *   ADCCR0<SGL><CNT>
  */
void ADC_Start(TSB_AD_TypeDef * ADx, ADC_TrgType Trg)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));
    assert_param(IS_ADC_START(Trg));

    if (Trg == ADC_TRG_SINGLE) {
        tmp = ADx->CR0 & ADC_MASK_SINGLE;
        tmp |= (uint32_t) (0x01U << 1U);
        ADx->CR0 = tmp;
    } else {                    /* ADC_TRG_CONSTANT */
        tmp = ADx->CR0 & ADC_MASK_CONTINUE;
        tmp |= (uint32_t) (0x01U << 0U);
        ADx->CR0 = tmp;
    }
}


/**
  * @brief  Get the conversion state of the specified ADC unit.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *   TSB_ADA.
  * @param  Trg: Select trigger type.
  *   This parameter can be one of the following values:
  *   ADC_TRG_SINGLE, ADC_TRG_CONTINUE
  * @retval ADC conversion state.
  *   The value returned can be one of the followings:
  *   DONE, BUSY.
  * @register The used register:
  *   ADCST
  */
WorkState ADC_GetConvertState(TSB_AD_TypeDef * ADx, ADC_TrgType Trg)
{
    WorkState ret = DONE;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));
    assert_param(IS_ADC_TRGTYPE(Trg));

    tmp = ADx->ST;
    if (Trg == ADC_TRG_SINGLE) {
        tmp &= (ADC_MASK_KEEP1BIT << 2U);       /* only keep bit2 */
    } else {
        tmp &= (ADC_MASK_KEEP1BIT << 3U);       /* only keep bit3 */
    }

    if (tmp) {
        ret = BUSY;
    } else {
        /* Do nothing */
    }

    return ret;
}

/**
  * @brief  Get result from the specified AD Conversion Result Register.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *     TSB_ADA.
  * @param  ResultREGx: Select which ADC result register will be read.
  *   This parameter can be one of the following values:
  *     ADC_REG0, ADC_REG1, ADC_REG2, ADC_REG3, ADC_REG4, ADC_REG5,
  *     ADC_REG6, ADC_REG7, ADC_REG8, ADC_REG9, ADC_REG10, ADC_REG11,
  *     ADC_REG12, ADC_REG13, ADC_REG14, ADC_REG15, ADC_REG16, ADC_REG17,
  *     ADC_REG18, ADC_REG19, ADC_REG20, ADC_REG21, ADC_REG22, ADC_REG23
  * @retval A union with AD result and 2 bits extra information.
  *         Refer to the members of ADC_Result in .h file for more detail usage.
  * @register The used registers:
  *   ADCREGn (n can be 0 to 23)   
  */
ADC_Result ADC_GetConvertResult(TSB_AD_TypeDef * ADx, ADC_REGx ResultREGx)
{
    ADC_Result result = { 0U };
    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));
    assert_param(IS_ADC_REGx(ResultREGx));

    /* read out result register wanted */
    /* only keep bit15 to bit0 */
    if (ADx == TSB_ADA) {
        result.All = (*(ADA_REGx_Address[ResultREGx])) & (uint32_t) 0xFFFFU;
	} else if (ADx == TSB_ADB) {
        result.All = (*(ADB_REGx_Address[ResultREGx])) & (uint32_t) 0xFFFFU;
	} else if (ADx == TSB_ADC) {
        result.All = (*(ADC_REGx_Address[ResultREGx])) & (uint32_t) 0xFFFFU;
    } else {
        result.All = 0;
    }

    return result;
}


/**
  * @brief  Set Software Trigger Program Register of the specified ADC unit.
  * @param  ADx: Select ADC unit
  *   This parameter can be one of the following values:
  *     TSB_ADA
  * @param  ResultREGx: Select which ADC result register will be used for the specified ADC unit.
  *   This parameter can be one of the following values:
  *     ADC_REG0, ADC_REG1, ADC_REG2, ADC_REG3, ADC_REG4, ADC_REG5,
  *     ADC_REG6, ADC_REG7, ADC_REG8, ADC_REG9, ADC_REG10, ADC_REG11,
  *     ADC_REG12, ADC_REG13, ADC_REG14, ADC_REG15, ADC_REG16, ADC_REG17,
  *     ADC_REG18, ADC_REG19, ADC_REG20, ADC_REG21, ADC_REG22, ADC_REG23
  * @param  MacroAINx: Select AD Channel together with its enabled or disabled setting.
  *   This parameter must be inputed with macro as the format below:
  *     TRG_ENABLE(x), TRG_DISABLE(x).
  *          ADC_AIN0, ADC_AIN1, ADC_AIN2, ADC_AIN3, ADC_AIN4,
  *          ADC_AIN5, ADC_AIN6, ADC_AIN7, ADC_AIN8, ADC_AIN9,
  *          ADC_AIN10, ADC_AIN11, ADC_AIN12, ADC_AIN13, ADC_AIN14,
  *          ADC_AIN15, ADC_AIN16, ADC_AIN17, ADC_AIN18
  * @retval None
  * @register The used registers:
  *   ADCTSETn (n can be 0 to 23) 
  */
void ADC_SetSWTrg(TSB_AD_TypeDef * ADx, ADC_REGx ResultREGx, uint8_t MacroAINx)
{
    /* Check the parameters */
    assert_param(IS_ADC_UNIT(ADx));
    assert_param(IS_ADC_REGx(ResultREGx));
    if (ADx == TSB_ADA) {
        assert_param(IS_UNIT_A_AINx(MacroAINx));
	} else if (ADx == TSB_ADB) {
        assert_param(IS_UNIT_B_AINx(MacroAINx));
    } else {
        assert_param(IS_UNIT_C_AINx(MacroAINx));
    }

    if (ADx == TSB_ADA) {
		*(ADA_TSETx_Address[ResultREGx]) = ((GEN_TRG_SINGLE << 5) | MacroAINx);
	} else if (ADx == TSB_ADB) {
		*(ADB_TSETx_Address[ResultREGx]) = ((GEN_TRG_SINGLE << 5) | MacroAINx);
	} else {
		*(ADC_TSETx_Address[ResultREGx]) = ((GEN_TRG_SINGLE << 5) | MacroAINx);
	}

}


/** @} */
/* End of group ADC_Exported_Functions */

/** @} */
/* End of group ADC */

/** @} */
/* End of group TX04_Periph_Driver */

