/**
 ****************************************************************************
 * @file	interrupt.c
 * @brief	Interrupt Functions for All Soft
 * @version	V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#include <stdlib.h>

#include "ipdefine.h"
#include "mcuip_drv.h"
#include "usercon.h"
#include "HPFC_drv.h"
#include "HPFC_para.h"
#include "D_Driver.h"
#define DEFINE_APP
#include "interrupt.h"
#undef  DEFINE_APP
/* #define MOTOR_OUTPUT  0 */
#define MOTOR_OUTPUT	1

/* for debug */
#if defined(__USE_VE_COMP) || defined(__USE_VE_FAN)
 #if defined(__MOTOR_DBGOUT_IDET)
	#define	DEBUG_IDET_ERROR_VE(x)		debug_idet_error_ve(x)
	void	debug_idet_error_ve(TSB_VE_TypeDef * const VEx)
	{
		if ((VEx->MCTLF & 0x0010) != 0)
		{
			TSB_PU_DATA_PU5 = 1U;
		}
		else
		{
			TSB_PU_DATA_PU5 = 0U;
		}
	}
 #else
	#define	DEBUG_IDET_ERROR_VE(x)
 #endif /* defined(__MOTOR_DBGOUT_IDET) */
#endif /* defined(__USE_VE_COMP) || defined(__USE_VE_FAN) */

#if defined(__CONTROL_MOTOR_COMP)&&!defined(__USE_VE_COMP)
 #if defined(__MOTOR_DBGOUT_IDET)
	#define	DEBUG_IDET_ERROR(x)			debug_idet_error(x)
	void	debug_idet_error(vector_t* const _motor)
	{
		if (_motor->drv.idetect_error != 0)
		{
			TSB_PU_DATA_PU5 = 1U;
		}
		else
		{
			TSB_PU_DATA_PU5 = 0U;
		}
	}
 #else
	#define	DEBUG_IDET_ERROR(x)
 #endif	/* defined(__MOTOR_DBGOUT_IDET) */
#endif /* defined(__CONTROL_MOTOR_CH1)&&!defined(__USE_VE_CH1) */

/**
 *********************************************************************************************
 * @brief		INT_interval_4kHz
 *
 * @return		none
 *********************************************************************************************
 */
extern uint8_t system_poweron;
extern uint16_t motor_error_cnt;
void INT_interval_4kHz(void)
{
	M_Main_Counter++;	   /* Count for main routine Period */
	if ((Hpfc.drv.state.flg.emg_ZC == 0) && (system_poweron == 0))
	{
		motor_error_cnt++;
	}
}

/**
 ********************************************************************************************
 * @brief		Vector Control Interruption by VE for ch0.
 *
 * @param[in]	vector_t* const			_motor	.
 * @param[in]	TSB_VE_TypeDef* const	VEx		.
 *
 * @return		none
 ********************************************************************************************
 */
#if defined(__CONTROL_MOTOR_FAN) && defined(__USE_VE_FAN)
void INT_VectorControlByVE_FAN(TSB_PMD_TypeDef* const PMDx, vector_t* const _motor, TSB_VE_TypeDef * const VEx)
{
#if defined(__CONTROL_PFC)
	pfc_t *const _pfc = &Hpfc;
#endif

	DEBUG_IDET_ERROR_VE(VEx);

#if (__FIXED_VDC_FAN == 1)								/* 1: Vdc from constant data */
	/***** Set Vdc data to VE *****/
	VE_SetVDCreg(VEx, (q15_t)(FIX_15 * cVDC_FAN / cV_MAX_FAN));
#endif

	/***** Get data from VE register *****/
	VE_GetdataFromVEreg(VEx, _motor);

	/***** Check over current *****/
	VE_GetPhaseCurrent(VEx, &_motor->drv.Ia, &_motor->drv.Ib, &_motor->drv.Ic);

	if (D_Check_OverCurrent(_motor->para.err_ovc, _motor->drv.Ia, _motor->drv.Ib, _motor->drv.Ic))
	{
		_motor->drv.state.flg.emg_S = SET;
		_motor->stage.itr = ciEmergency;
	}

	/***** Check error voltage *****/
	if ((_motor->drv.Vdc < (q15_t)(FIX_15 * cVDC_MINLIM_FAN / cV_MAX_FAN)) || ((q15_t)(FIX_15 * cVDC_MAXLIM_FAN / cV_MAX_FAN) < _motor->drv.Vdc))
	{
		_motor->stage.itr = ciEmergency;
	}
	_motor->drv.Vdq = VE_GET_Cal_Vdq(_motor->drv.Vd.half[1], _motor->drv.Vq.half[1]);
	if (_motor->drv.state.all == 0)						/* No error? */
	{
		/***** Detect Rotor Position *****/
		D_Detect_Rotor_Position(_motor);

#if defined(__USE_ENCODER_FAN)
		/***** Encoder *****/
		H_Encoder(_motor, TSB_EN0);
#endif /* __USE_ENCODER_FAN */

		/***** Control Speed *****/
		_motor->drv.spdprd++;
		if (_motor->drv.spdprd >= cSPD_PI_PRD_FAN)
		{
			_motor->drv.spdprd = 0;
			D_Control_Speed(_motor);
		}
	}

	/***** Check Current detection error  *****/
#if defined(__Debug__)
	_motor->drv.DutyU = VEx->CMPU;
	_motor->drv.DutyV = VEx->CMPV;
	_motor->drv.DutyW = VEx->CMPW;
	D_Check_DetectCurrentError(_motor);
#endif

	/**** Set data to VE register *****/
	/* Set Shift pwm change speed. for 1shunt */
	VE_SetSpwmMode(VEx, _motor->drv.command.spwm);

	/* Set modulation type */
	VE_SetModulType(VEx, _motor->drv.command.modul);

	switch (_motor->stage.itr)
	{
		/*--- Stop Stage -----------------------------*/
		case ciStop:
			VE_SetdataToVEreg_Stop(VEx, _motor);
			break;

		/*--- Bootstrap Stage ------------------------*/
		case ciBootstrap:
			VE_SetdataToVEreg_Bootstrap(VEx, _motor);
			break;

		/*--- Position Stage (Current type) ----------*/
		case ciInitposition_i:
			VE_SetdataToVEreg_Initposition_i(VEx, _motor);
			break;

		/*--- Position Stage (Voltage type) ----------*/
		case ciInitposition_v:
			VE_SetdataToVEreg_Initposition_v(VEx, _motor);
			break;

		/*--- Force Stage (Current type) -------------*/
		case ciForce_i:
			VE_SetdataToVEreg_Force_i(VEx, _motor);
			break;

		/*--- Force Stage (Voltage type) -------------*/
		case ciForce_v:
			VE_SetdataToVEreg_Force_v(VEx, _motor);
			break;

		/*--- ChangeUp Stage -------------------------*/
		case ciChange_up:
			VE_SetdataToVEreg_Change_up(VEx, _motor);
			break;

		/*--- Steady_A Stage ---------------------------*/
		case ciSteady_A:
			VE_SetdataToVEreg_Steady_A(PMDx, VEx, _motor);
			break;

		/*--- Emergency Stage ------------------------*/
		case ciEmergency:
			VE_SetdataToVEreg_Emergency(VEx, _motor);
			break;

		/*--- Fail-safe ------------------------*/
		default:
			VE_SetdataToVEreg_Stop(VEx, _motor);
			break;
	}

#if defined(__USE_DAC)
	UiOutDataStartByVE(0);							/* Dac output for debug */
#endif /* __USE_DAC */

	VE_Start(VEx);

#if defined(__CONTROL_PFC)
	HPFC_VacVdc_Handle(_pfc);
#endif
}
#endif /* defined(__CONTROL_MOTOR_FAN) && defined(__USE_VE_FAN) */

/**
 ********************************************************************************************
 * @brief	   Vector Control Interruption for ch1. (All software control.)
 *
 * @param[in]   vector_t* const		 _motor  .
 * @param[in]   TSB_PMD_TypeDef* const  PMDx	.
 *
 * @return	  none
 ********************************************************************************************
 */
#if defined(__CONTROL_MOTOR_COMP) && !defined(__USE_VE_COMP)
void INT_VectorControlBySoft_COMP(vector_t* const _motor, TSB_PMD_TypeDef* const PMDx)
{
#if defined(__CONTROL_PFC)
	pfc_t *const _pfc = &Hpfc;
#endif

#if defined(__CONTROL_PFC)
	if ((_motor->stage.main == cSteady_A)||(_motor->stage.main == cForce))
	{
		D_Power_Limit(_pfc, _motor);
	}
#endif

	/***** Get data from ADC register *****/
	D_GetMotorCurrentPowerVolt(_motor);
	_motor->drv.Vdc = Motor_fan.drv.Vdc;

#if defined(__CONTROL_PFC)
	/***************PFC****************/
	if (_pfc->pfc_signal.update_pfc)
	{
		_pfc->pfc_signal.update_pfc = 0;
		if (_pfc->pfc_signal.status == 1)
		{
			HPFC_PWMOutputOn_pre(_pfc);
		}
		else if (_pfc->pfc_signal.status == 2)
		{
			HPFC_PWMOutputOn(_pfc);
		}
		else if (_pfc->pfc_signal.status == 3)
		{
			HPFC_PWMOutputOff(_pfc);
		}
	}
#endif

	_motor->drv.Ia_t += (_motor->drv.Ia - _motor->drv.Ia_t) >> 3;
	_motor->drv.Ib_t += (_motor->drv.Ib - _motor->drv.Ib_t) >> 3;
	_motor->drv.Ic_t += (_motor->drv.Ic - _motor->drv.Ic_t) >> 3;

	/***** Check over current *****/
	if (D_Check_OverCurrent(_motor->para.err_ovc, _motor->drv.Ia_t, _motor->drv.Ib_t, _motor->drv.Ic_t))
	{
		_motor->drv.state.flg.emg_S = SET;
		_motor->stage.itr = ciEmergency;
	}

	if (_motor->drv.state.all == 0)						/* No error? */
	{
		/***** Input phase transformation *****/
		D_InputTransformation(_motor);

		_motor->drv.Vdq = VE_GET_Cal_Vdq(_motor->drv.Vd.half[1], _motor->drv.Vq.half[1]);

		_motor->drv.Vdq_ave.word = calc_ave(_motor->drv.Vdq_ave.word, _motor->drv.Vdq << 16, cVDQ_AVE);
#if defined(__CONTROL_PFC)
		_pfc->judgevac.Vdq = _motor->drv.Vdq_ave.half[1] * cPFC_VDQ_PERCENT / 100;

		if (++_pfc->judgevac.cnt_vdq < cPFC_VDQ_CNT)
		{
			_pfc->judgevac.Vdq_sum += _pfc->judgevac.Vdq;
		}
		else
		{
			_pfc->judgevac.Vdq_ave = _pfc->judgevac.Vdq_sum / cPFC_VDQ_CNT;
			_pfc->judgevac.Vdq_sum = 0;
			_pfc->judgevac.cnt_vdq = 0;
		}
#endif

		/***** Detect Rotor Position *****/
		D_Detect_Rotor_Position(_motor);

#if defined(__USE_ENCODER_COMP)
		/***** Encoder *****/
		H_Encoder(_motor, TSB_EN1);
#endif /* __USE_ENCODER_FAN */

		/***** Control Speed *****/
		_motor->drv.spdprd++;
		if (_motor->drv.spdprd >= cSPD_PI_PRD_COMP)
		{
			_motor->drv.spdprd = 0;
			D_Control_Speed(_motor);
		}

		/***** Check Current detection error  *****/
		D_Check_DetectCurrentError(_motor);
		DEBUG_IDET_ERROR(_motor);

		/***** Current control  *****/
		D_Control_Current(_motor);

		/***** Output coordinate axis transformation  *****/
		D_OutputTransformation(_motor);

		/***** Trigger generation  *****/
		D_CalTrgTiming(PMDx, _motor);
	}

	/***** Output control  *****/
	PMD_SetTriggerSelect(PMDx, _motor->drv.Sector1);
	PMD_RegDataSet(PMDx, (uint16_t)_motor->drv.DutyU, (uint16_t)_motor->drv.DutyV, (uint16_t)_motor->drv.DutyW, (q15_t)_motor->drv.AdTrg0, (q15_t)_motor->drv.AdTrg1);

#if MOTOR_OUTPUT == 1
	D_DecisionOutputMode(_motor, PMDx);
#endif

#if defined(__USE_DAC)
	/* UiOutDataStart(1); */									/* Dac output for debug */
#endif /* __USE_DAC */

}
#endif /* defined(__CONTROL_MOTOR_COMP) && !defined(__USE_VE_COMP) */

/*********************************** END OF FILE ******************************/
