/**
 ****************************************************************************
 * @file    ipdrv_uart.c
 * @brief   This file provides API functions for UART driver.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "ipdrv_uart.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @defgroup UART
  * @brief UART driver modules
  * @{
  */

/** @defgroup UART_Private_Defines
  * @{
  */


#define UART_ERROR_MASK        ((uint32_t)0x0000001F)
#define UART_OERR_FLAG         ((uint8_t)0x08)
#define UART_PERR_FLAG         ((uint8_t)0x04)
#define UART_FERR_FLAG         ((uint8_t)0x02)
#define UART_TRIGGER_FLAG      ((uint8_t)0x10)
#define UART_BREAK_FLAG        ((uint8_t)0x01)

#define CR0_PARITY_MASK        ((uint32_t)0x0000F773)
#define CR0_CTSE_MASK          ((uint32_t)0x0000F37F)
#define CR0_LPB_SET            ((uint32_t)0x00008000)
#define CR0_LPB_CLEAR          ((uint32_t)0x0000777F)
#define CR0_NF_MASK            ((uint32_t)0x0000877F)
#define CR0_RTSE_MASK          ((uint32_t)0x0000F57F)
#define CR0_RTSE_SET           ((uint32_t)0x00000200)
#define CR0_CTSE_SET           ((uint32_t)0x00000400)
#define CR0_IV_MASK            ((uint32_t)0x0000F73F)
#define CR0_DIR_MASK           ((uint32_t)0x0000F75F)
#define CR0_SBLEN_MASK         ((uint32_t)0x0000F76F)
#define CR0_BIT_18_16_CLEAR    ((uint32_t)0x0000F77F)
#define CR0_WU_SET             ((uint32_t)0x00000100)
#define CR0_WU_CLEAR           ((uint32_t)0x0000F67F)
#define CR0_SM_MASK            ((uint32_t)0x0000F77C)

#define CLK_CKSEL_MASK         ((uint32_t)0x000000F0)
#define CLK_CKSEL_BRG          ((uint32_t)0x00000000)
#define CLK_PRSEL_MASK         ((uint32_t)0x00000003)

#define TRANS_RXE_CLEAR        ((uint32_t)0x0000000E)
#define TRANS_TXE_CLEAR        ((uint32_t)0x0000000D)
#define TRANS_BK_MASK          ((uint32_t)0x00000007)
#define TRANS_TXTRG_MASK       ((uint32_t)0x0000000B)
#define TRANS_TXTRG_SET        ((uint32_t)0x00000004)

#define SWRST_MASK             ((uint32_t)0x00000080)
#define SWRST_CMD1             ((uint32_t)0x00000002)
#define SWRST_CMD2             ((uint32_t)0x00000001)

#define BRD_KEN_SET            ((uint32_t)0x00800000)
#define BRD_BRN_MASK           ((uint32_t)0x00BF0000)
#define BRD_BRK_MASK           ((uint32_t)0x0080FFFF)

#define CR1_INTTXFE_SET        ((uint32_t)0x00000080)
#define CR1_INTTXFE_CLEAR      ((uint32_t)0x00007777)
#define CR1_INTRXFE_SET        ((uint32_t)0x00000020)
#define CR1_INTRXFE_CLEAR      ((uint32_t)0x000077D7)
#define CR1_INTRXWE_MASK       ((uint32_t)0x000077E7)
#define CR1_INTRXWE_SET        ((uint32_t)0x00000010)
#define CR1_INTTXWE_MASK       ((uint32_t)0x000077B7)
#define CR1_INTTXWE_SET        ((uint32_t)0x00000040)
#define CR1_INTERR_MASK        ((uint32_t)0x000077F3)
#define CR1_INTERR_SET         ((uint32_t)0x00000004)
#define CR1_DMARE_MASK         ((uint32_t)0x000077F6)
#define CR1_DMARE_SET          ((uint32_t)0x00000001)
#define CR1_DMATE_MASK         ((uint32_t)0x000077F5)
#define CR1_DMATE_SET          ((uint32_t)0x00000002)
#define CR1_RIL_CLEAR          ((uint32_t)0x000070F7)
#define CR1_TIL_CLEAR          ((uint32_t)0x000007F7)

#define FIFOCLR_TFCLR_SET      ((uint32_t)0x00000002)
#define FIFOCLR_RFCLR_SET      ((uint32_t)0x00000001)

#define SR_TLVL_MASK           ((uint32_t)0x00000F00)
#define SR_RLVL_MASK           ((uint32_t)0x0000000F)
#define SR_SUE_MASK            ((uint32_t)0x80000000)
#define SR_SUE_CLEAR           ((uint32_t)0x00000000)
#define SR_TXEND_MASK          ((uint32_t)0x00004000)
#define SR_RXEND_MASK          ((uint32_t)0x00000040)
#define SR_TXFF_MASK           ((uint32_t)0x00002000)
#define SR_RXFF_MASK           ((uint32_t)0x00000020)

#define ERR_OVRERR_CLEAR       ((uint32_t)0x00000017)
#define ERR_PERR_CLEAR         ((uint32_t)0x0000001B)
#define ERR_FERR_CLEAR         ((uint32_t)0x0000001D)
#define ERR_TRGERR_CLEAR       ((uint32_t)0x0000000F)
#define ERR_BERR_CLEAR         ((uint32_t)0x0000001E)
#define ERR_ALLERR_CLEAR       ((uint32_t)0x00000000)

/** @} */
/* End of group UART_Private_Defines */

/** @defgroup UART_Private_FunctionPrototypes
  * @{
  */

/** @} */
/* End of group UART_Private_FunctionPrototypes */

/** @defgroup UART_Private_Functions
  * @{
  */

/** @} */
/* End of group UART_Private_Functions */

/** @defgroup UART_Exported_Functions
  * @{
  */

 /**
  * @brief Get Status Enable Flag of the specified UART channel.
  * @param UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval Status of Enable Flag
  *   The value returned can be one of the followings:
  *   PROGRAMMABLE, NOT_PROGRAMMABLE.
  */
Register_Flag UART_GetStatusEnableFlag(TSB_UART_TypeDef * UARTx )
{
    Register_Flag retval = NOT_PROGRAMMABLE;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    if ((UARTx->SR & SR_SUE_MASK) == SR_SUE_CLEAR) {
        retval = PROGRAMMABLE;
    }
    return retval;
}

/**
  * @brief  Reset the specified UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval None
  */
void UART_SWReset(TSB_UART_TypeDef * UARTx)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    /* Write software-reset command */
    tmp = UARTx->SWRST;
    tmp &= SWRST_MASK;
    UARTx->SWRST = tmp | SWRST_CMD1;
    tmp &= SWRST_MASK;
    UARTx->SWRST = tmp | SWRST_CMD2;
}

/**
  * @brief  Initialize the specified UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  InitStruct: The structure containing basic UART configuration.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_Init(TSB_UART_TypeDef * UARTx, UART_InitTypeDef * InitStruct)
{
    Result retval = ERROR;
    uint32_t T0 = 0U;
    uint32_t T = 0U;
    uint32_t t = 1U;
    uint32_t N = 0U;
    uint32_t K = 0U;
    uint32_t tmp = 0U;
    uint32_t divider = 0U;
    const uint32_t a = 1U;

    /* Check the parameters */
    assert_param(IS_POINTER_NOT_NULL(InitStruct));
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_BAUDRATE(InitStruct->BaudRate));
    assert_param(IS_UART_DATA_BITS(InitStruct->DataBits));
    assert_param(IS_UART_DATA_INVERT(InitStruct->DataInvert));
    assert_param(IS_UART_TRANSFER_DIR(InitStruct->TransferDirection));
    assert_param(IS_UART_STOPBITS(InitStruct->StopBits));
    assert_param(IS_UART_PARITY(InitStruct->Parity));
    assert_param(IS_UART_FLOW_CONTROL(InitStruct->FlowCtrl));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {

        /* Use baud rate generator */
        tmp = UARTx->CLK;
        tmp &= CLK_CKSEL_MASK;
        tmp |= CLK_CKSEL_BRG;
        UARTx->CLK = tmp;

        /* Configure the flow control */
        tmp = UARTx->CR0;
        tmp &= CR0_SM_MASK;
        tmp &= CR0_CTSE_MASK;
        tmp |= (InitStruct->DataBits | InitStruct->FlowCtrl);

        /* Set Data signal invert function */
        tmp &= CR0_IV_MASK;
        tmp |= InitStruct->DataInvert;

        /* Set Data transfer direction*/
        tmp &= CR0_DIR_MASK;
        tmp |= InitStruct->TransferDirection;

        /* Set the stop bit */
        tmp &= CR0_SBLEN_MASK;
        tmp |= InitStruct->StopBits;

        /* Enable or disable parity check */
        tmp &= CR0_PARITY_MASK;
        tmp |= InitStruct->Parity;

        tmp &= CR0_BIT_18_16_CLEAR;
        UARTx->CR0 = tmp;

        /* Get the peripheral I/O clock frequency */
        SystemCoreClockUpdate();
		T0 = (SystemCoreClock / (a << ((TSB_CG->SYSCR >> 8U) & 7U))) >> 1; /* 80M */

        /* Baud rate setting */
        while ((divider < 100U) || (divider > 6553500U)) {
            T = a << t;
            divider = (100ULL * T0) / (16U * InitStruct->BaudRate * T);
            t++;
        }

        /* Select division ratio of the prescaler */
        tmp = UARTx->CLK;
        tmp &= CLK_PRSEL_MASK;
        tmp |= (((t - 1U) & 0xFU) << 4U);
        UARTx->CLK |= tmp;

        /* Set N +(64-K)/64 division and set N-value */
        N = divider / 100U;
        tmp = UARTx->BRD;
        if ((divider - (N * 100U)) == 0U) {
            tmp &= ~BRD_KEN_SET;
        } else {
            tmp |= BRD_KEN_SET;
        }
        tmp &= BRD_BRN_MASK;
        tmp |= (N & 0x0FFFFU);
        UARTx->BRD = tmp;

        /* Set K-value of N +(64-K)/64 division */
        if ((divider - (N * 100U)) == 0U) {
            /* Do nothing */
        } else {
            K = (64U * (100U - (divider - (N * 100U)))) / 100U;
            if (K < 1U) {
                K = 1U;
            } else {
                /* Do nothing */
            }
            tmp = UARTx->BRD;
            tmp &= BRD_BRK_MASK;
            tmp |= ((K << 16) & 0x3F0000U);
            UARTx->BRD = tmp;
        }

        tmp = UARTx->TRANS;
        /* Enable or disable transmission or reception */
        switch (InitStruct->Mode) {
        case UART_ENABLE_RX:
            tmp |= InitStruct->Mode;
            tmp &= TRANS_TXE_CLEAR;
            break;
        case UART_ENABLE_TX:
            tmp |= InitStruct->Mode;
            tmp &= TRANS_RXE_CLEAR;
            break;
        default:
            tmp |= UART_ENABLE_RX;
            tmp |= UART_ENABLE_TX;
            break;
        }
        UARTx->TRANS = tmp;
        retval = SUCCESS;

    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Get received data of the specified UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval The received data
  */
uint32_t UART_GetRxData(TSB_UART_TypeDef * UARTx)
{
    uint32_t retval = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    /* Return received data */
    retval = retval | (UARTx->DR & 0x1FFU);
    return retval;
}

/**
  * @brief  Set data to be sent and start transmitting via the specified
      UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  Data: the data to be sent. It is between 0x000 and 0x1FF.
  * @retval None
  */
void UART_SetTxData(TSB_UART_TypeDef * UARTx, uint32_t Data)
{
    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_DATA(Data));

    UARTx->DR = Data & 0x1FFU;
}

/**
  * @brief  Initialize the specified UART channel in default configuration.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_DefaultConfig(TSB_UART_TypeDef * UARTx)
{
    Result retval = ERROR;
    UART_InitTypeDef uartdefault;
    uartdefault.BaudRate = 115200U;
    uartdefault.DataBits = UART_DATA_BITS_8;
    uartdefault.DataInvert = UART_NO_DATA_INVERT;
    uartdefault.TransferDirection = UART_LSB;
    uartdefault.StopBits = UART_STOP_BITS_1;
    uartdefault.Parity = UART_NO_PARITY;
    uartdefault.Mode = UART_ENABLE_RX | UART_ENABLE_TX;
    uartdefault.FlowCtrl = UART_NONE_FLOW_CTRL;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    /* Select baud rate generator as UART source clcok */
    /* Set baud rate as 115200bps */
    /* Select 8-bit UART mode */
    /* Select 1-bit stop */
    /* No parity check */
    /* No flow control */
    /* Enable both transmission and reception */
    if(UART_Init(UARTx, &uartdefault) == SUCCESS) {
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Indicate UART transfer error.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval The error flag.
  *   The value returned can be one of the followings:
  *   UART_NO_ERR, UART_OVERRUN, UART_PARITY_ERR, UART_FRAMING_ERR , UART_TRIGGER_ERR, UART_BREAK_ERR or UART_ERRS.
  */
UART_Err UART_GetErrState(TSB_UART_TypeDef * UARTx)
{
    uint8_t tmp = 0U;
    UART_Err retval = UART_NO_ERR;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    tmp = ((uint8_t) (UARTx->ERR & UART_ERROR_MASK));
    switch (tmp) {
    case UART_OERR_FLAG:         /* Check overrun flag */
        retval = UART_OVERRUN;
        break;
    case UART_PERR_FLAG:         /* Check parity flag */
        retval = UART_PARITY_ERR;
        break;
    case UART_FERR_FLAG:         /* Check framing flag */
        retval = UART_FRAMING_ERR;
        break;
    case UART_TRIGGER_FLAG:       /* Check trigger flag */
        retval = UART_TRIGGER_ERR;
        break;
    case UART_BREAK_FLAG:         /* Check break flag */
        retval = UART_BREAK_ERR;
        break;
    default:
        if (tmp != 0U) {
            /* more than one error */
            retval = UART_ERRS;
        } else {
      		retval = UART_NO_ERR;
        }
        break;
    }
    return retval;
}

/**
  * @brief  Clear UART transfer error.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  UARTx: Select the UART error type.
  *   This parameter can be one of the following values:
  *   UART_OVERRUN_TYPE, UART_PARITY_ERR_TYPE, UART_FRAMING_ERR_TYPE, UART_TRIGGER_ERR_TYPE,
  *   UART_BREAK_ERR_TYPE or UART_ALL_ERR_TYPE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_ClearErrState(TSB_UART_TypeDef * UARTx, uint8_t ErrorType)
{
    Result retval = ERROR;
    uint8_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_ERROR(ErrorType));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->ERR;
        switch (ErrorType) {
        case UART_OVERRUN_TYPE:           /* clear overrun error */
            tmp &= ERR_OVRERR_CLEAR;
            break;
        case UART_PARITY_ERR_TYPE:         /* clear parity error */
            tmp &= ERR_PERR_CLEAR;
            break;
        case UART_FRAMING_ERR_TYPE:       /* clear framing error */
            tmp &= ERR_FERR_CLEAR;
            break;
        case UART_TRIGGER_ERR_TYPE:       /* Clear trigger error */
            tmp &= ERR_TRGERR_CLEAR;
            break;
        case UART_BREAK_ERR_TYPE:         /* Clear break error */
            tmp &= ERR_BERR_CLEAR;
            break;
        default:
            tmp &= ERR_ALLERR_CLEAR;  /* Clear all error */
        }
        UARTx->ERR = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable the wake-up function in 9-bit UART mode
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of this function.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_SetWakeUpFunc(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{

    Result retval = ERROR;
    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        if (NewState == ENABLE) {
            /* Set SCxCR0<WU> to enable wake-up function */
            UARTx->CR0 &= CR0_BIT_18_16_CLEAR;
            UARTx->CR0 |= CR0_WU_SET;
        } else {
            /* Clear SCxCR0<WU> to disable wake-up function */
            UARTx->CR0 &= CR0_BIT_18_16_CLEAR;
            UARTx->CR0 &= CR0_WU_CLEAR;
        }
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable Loop Back Function.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of this function
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */

Result UART_SetLoopBackFunc(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR0;
        if (NewState == ENABLE) {
            /* Set SCxCR0<LPB> to enable Loop-back function */
            tmp &= CR0_BIT_18_16_CLEAR;
            UARTx->CR0 = tmp | CR0_LPB_SET;
        } else {
            /* Clear SCxCR0<LPB> to enable Loop-back function */
            tmp &= CR0_BIT_18_16_CLEAR;
            UARTx->CR0= tmp & CR0_LPB_CLEAR;
        }
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Set noise eliminate time for UTxRX.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  Elimination_Time: eliminate time for UTxRx.
  *   This parameter can be one of the following values:
  *   ELIMINATION_TIME_NONE, ELIMINATION_TIME_2_PHIT0,
  *   ELIMINATION_TIME_4_PHIT0, ELIMINATION_TIME_8_PHIT0,
  *   ELIMINATION_TIME_2_TRANSFERCLK, ELIMINATION_TIME_3_TRANSFERCLK,
  *   ELIMINATION_TIME_4_TRANSFERCLK, ELIMINATION_TIME_5_TRANSFERCLK,
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_SetNoiseEliminateTime(TSB_UART_TypeDef * UARTx, uint32_t Elimination_Time)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_NOISE_ELIMINATION_TIME(Elimination_Time));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR0;
        tmp &= CR0_NF_MASK;
        tmp &= CR0_BIT_18_16_CLEAR;
        UARTx->CR0 = tmp | Elimination_Time;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Set break transmission for UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  breakmode: Break mode of transmission for UART channel.
  *   This parameter can be: UART_NO_TRANSMISSION or UART_TRANSMISSION.
  * @retval None
  */
void UART_SetBreakTransmission(TSB_UART_TypeDef * UARTx, uint32_t breakmode)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_BREAK_TRANSMISSION(breakmode));

    tmp = UARTx->TRANS;
    tmp &= TRANS_BK_MASK;
    tmp |= breakmode;
    UARTx->TRANS = tmp;
}

/**
  * @brief  Set trigger transmission for UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of this function
  *   This parameter can be: ENABLE or DISABLE.
  * @retval None
  */
void UART_SetTriggerTransmission(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    tmp = UARTx->TRANS;
    tmp &= TRANS_TXTRG_MASK;
    if (NewState == ENABLE) {
            tmp |= TRANS_TXTRG_SET;
            tmp &= TRANS_TXE_CLEAR;
    } else {
            /* Do nothing */
    }
    UARTx->TRANS = tmp;
}


/**
  * @brief  Enable or disable handshake function in reception for UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of this function
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_RxHandshakeFunc(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR0;
        tmp &= CR0_RTSE_MASK;
        if (NewState == ENABLE) {
            tmp |= CR0_RTSE_SET;
        }else {
            /* Do nothing */
        }
        tmp &= CR0_BIT_18_16_CLEAR;
        UARTx->CR0 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable handshake function in transmit for UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of this function
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_TxHandshakeFunc(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR0;
        tmp &= CR0_CTSE_MASK;
        if (NewState == ENABLE) {
            tmp |= CR0_CTSE_SET;
        }else {
            /* Do nothing */
        }
        tmp &= CR0_BIT_18_16_CLEAR;
        UARTx->CR0 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable receive interrupt for receive FIFO.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of receive interrupt for receive FIFO.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_RxFIFOINTCtrl(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        if (NewState == ENABLE) {
            /* Set SCxCR1<INTRXFE> to enable receive FIFO interrupt */
            UARTx->CR1 = tmp | CR1_INTRXFE_SET;
        } else {
            /* Clear SCxCR1<INTRXFE> to disable receive FIFO interrupt */
            UARTx->CR1 = tmp & CR1_INTRXFE_CLEAR;
        }
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable transmit interrupt for transmit FIFO.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *    UART0, UART1, UART2, UART3.
  * @param  NewState: New state of transmit interrupt for transmit FIFO.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_TxFIFOINTCtrl(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        if (NewState == ENABLE) {
            /* Set SCxCR1<INTTXFE> to enable transmit FIFO interrupt */
            UARTx->CR1 = tmp | CR1_INTTXFE_SET;
        } else {
            /* Clear SCxCR1<INTTXFE> to disable transmit FIFO interrupt */
            UARTx->CR1 = tmp & CR1_INTTXFE_CLEAR;
        }
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Receive FIFO fill level to generate receive interrupts.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  RxFIFOLevel: Receive FIFO fill level.
  *   This parameter can be one of the following values:
  *   UART_RXFIFO_FLEVLE_8B, UART_RXFIFO_FLEVLE_1B,
  *   UART_RXFIFO_FLEVLE_2B, UART_RXFIFO_FLEVLE_3B,
  *   UART_RXFIFO_FLEVLE_4B, UART_RXFIFO_FLEVLE_5B,
  *   UART_RXFIFO_FLEVLE_6B, UART_RXFIFO_FLEVLE_7B.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_RxFIFOFillLevel(TSB_UART_TypeDef * UARTx, uint32_t RxFIFOLevel)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_RXFIFO_FLEVLE(RxFIFOLevel));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        tmp &= CR1_RIL_CLEAR;
        tmp |= RxFIFOLevel;
        UARTx->CR1 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Transmit FIFO fill level to generate transmit interrupts.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  TxFIFOLevel: Transmit FIFO fill level.
  *   This parameter can be one of the following values:
  *   UART_TXFIFO_FLEVLE_0B, UART_TXFIFO_FLEVLE_1B,
  *   UART_TXFIFO_FLEVLE_2B, UART_TXFIFO_FLEVLE_3B,
  *   UART_TXFIFO_FLEVLE_4B, UART_TXFIFO_FLEVLE_5B,
  *   UART_TXFIFO_FLEVLE_6B, UART_TXFIFO_FLEVLE_7B,
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_TxFIFOFillLevel(TSB_UART_TypeDef * UARTx, uint32_t TxFIFOLevel)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_TXFIFO_FLEVLE(TxFIFOLevel));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        tmp &= CR1_TIL_CLEAR;
        tmp |= TxFIFOLevel;
        UARTx->CR1 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Receive FIFO clear.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_RxFIFOClear(TSB_UART_TypeDef * UARTx)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->FIFOCLR;
        tmp |= FIFOCLR_RFCLR_SET;
        UARTx->FIFOCLR = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Transmit FIFO clear.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_TxFIFOClear(TSB_UART_TypeDef * UARTx)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->FIFOCLR;
        tmp |= FIFOCLR_TFCLR_SET;
        UARTx->FIFOCLR = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable receive interrupt for receive UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of receive interrupt for UART channel.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_RxINTCtrl(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        tmp &= CR1_INTRXWE_MASK;
        if (NewState == ENABLE) {
            tmp |= CR1_INTRXWE_SET;
        }else {
             /* Do nothing */
        }
        UARTx->CR1 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable transmit interrupt for receive UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of transmit interrupt for UART channel.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_TxINTCtrl(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        tmp &= CR1_INTTXWE_MASK;
        if (NewState == ENABLE) {
            tmp |= CR1_INTTXWE_SET;
        }else {
             /* Do nothing */
        }
        UARTx->CR1 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable error interrupt for UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of error interrupt for UART channel.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_ErrINTCtrl(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        tmp &= CR1_INTERR_MASK;
        if (NewState == ENABLE) {
            tmp |= CR1_INTERR_SET;
        }else {
            /* Do nothing */
        }
        UARTx->CR1 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable receive DMA for UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of receive DMA for UART channel.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_RxDMACtrl(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        tmp &= CR1_DMARE_MASK;
        if (NewState == ENABLE) {
            tmp |= CR1_DMARE_SET;
        }else {
             /* Do nothing */
        }
        UARTx->CR1 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Enable or disable transmit DMA for UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  NewState: New state of transmit DMA for UART channel.
  *   This parameter can be: ENABLE or DISABLE.
  * @retval The value returned can be one of the following values:
  *   SUCCESS or ERROR
  */
Result UART_TxDMACtrl(TSB_UART_TypeDef * UARTx, FunctionalState NewState)
{
    Result retval = ERROR;
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (UART_GetStatusEnableFlag(UARTx) == PROGRAMMABLE) {
        tmp = UARTx->CR1;
        tmp &= CR1_DMATE_MASK;
        if (NewState == ENABLE) {
            tmp |= CR1_DMATE_SET;
        }else {
            /* Do nothing */
        }
        UARTx->CR1 = tmp;
        retval = SUCCESS;
    } else {
        /* Do nothing */
    }

    return retval;
}

/**
  * @brief  Get status transmit or reception of UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  Transfer_Mode: Transfer Mode of UART channel
  *   This parameter can be: UART_RECEPTION or UART_TRANSMIT.
  * @retval status of transfer
  * The value returned can be IN_PROGRESS, COMPLETED.
  */
Transfer_Status UART_GetTransferStatus(TSB_UART_TypeDef * UARTx, Uart_TransferMode Transfer_Mode)
{
    Transfer_Status retval = IN_PROGRESS;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_TRANSFER_MODE(Transfer_Mode));

    if (Transfer_Mode == UART_RECEPTION) {
        if ((UARTx->SR & SR_RXEND_MASK) == SR_RXEND_MASK) {
            retval = COMPLETED;
        }else {
             /* Do nothing */
        }
    } else {
        if ((UARTx->SR & SR_TXEND_MASK) == SR_TXEND_MASK) {
            retval = COMPLETED;
        }else {
             /* Do nothing */
        }
    }
    return retval;
}

/**
  * @brief  Get status FIFO transmit or reception of UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  Transfer_Mode: Transfer Mode of UART channel
  *   This parameter can be: UART_RECEPTION or UART_TRANSMIT.
  * @retval FIFO status of transmission
  * The value returned can be NO_REACH_FLEVEL, REACH_FLEVEL.
  */
FIFO_Status UART_GetFIFOStatus(TSB_UART_TypeDef * UARTx, Uart_TransferMode Transfer_Mode)
{
    FIFO_Status retval = NO_REACH_FLEVEL;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_TRANSFER_MODE(Transfer_Mode));

    if (Transfer_Mode == UART_RECEPTION) {
        if ((UARTx->SR & SR_RXFF_MASK) == SR_RXFF_MASK) {
            retval = REACH_FLEVEL;
        }else {
             /* Do nothing */
        }
    } else {
        if ((UARTx->SR & SR_TXFF_MASK) == SR_TXFF_MASK) {
            retval = REACH_FLEVEL;
        }else {
            /* Do nothing */
        }
    }
    return retval;
}

/**
  * @brief  Status of receive FIFO fill level.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval Receive FIFO fill level status.
  */
uint32_t UART_GetRxFIFOFillLevelStatus(TSB_UART_TypeDef * UARTx)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    tmp = UARTx->SR;
    tmp &= SR_RLVL_MASK;
    /* Return the value */
    return tmp;
}

/**
  * @brief  Status of transmit FIFO fill level.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @retval Transmit FIFO fill level status.
  */
uint32_t UART_GetTxFIFOFillLevelStatus(TSB_UART_TypeDef * UARTx)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));

    tmp = UARTx->SR;
    tmp &= SR_TLVL_MASK;
    /* Return the value */
    return tmp;
}

/**
  * @brief  Clear status transmit or reception of UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  Transfer_Mode: Transfer Mode of UART channel
  *   This parameter can be: UART_RECEPTION or UART_TRANSMIT.
  * @retval None
  *
  */
void UART_ClearTransferStatus(TSB_UART_TypeDef * UARTx, Uart_TransferMode Transfer_Mode)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_TRANSFER_MODE(Transfer_Mode));

    tmp = UARTx->SR;
    if (Transfer_Mode == UART_RECEPTION) {
        tmp |= SR_RXEND_MASK;
    } else {
        tmp |= SR_TXEND_MASK;
    }
    UARTx->SR = tmp;
}

/**
  * @brief  Clear FIFO status transmit or reception of UART channel.
  * @param  UARTx: Select the UART channel.
  *   This parameter can be one of the following values:
  *   UART0, UART1, UART2, UART3.
  * @param  Transfer_Mode: Transfer Mode of UART channel
  *   This parameter can be: UART_RECEPTION or UART_TRANSMIT.
  * @retval None
  *
  */
void UART_ClearFIFOStatus(TSB_UART_TypeDef * UARTx, Uart_TransferMode Transfer_Mode)
{
    uint32_t tmp = 0U;

    /* Check the parameters */
    assert_param(IS_UART_PERIPH(UARTx));
    assert_param(IS_UART_TRANSFER_MODE(Transfer_Mode));

    tmp = UARTx->SR;
    if (Transfer_Mode == UART_RECEPTION) {
        tmp |= SR_RXFF_MASK;
    } else {
        tmp |= SR_TXFF_MASK;
    }
    UARTx->SR = tmp;
}

/** @} */
/* End of group UART_Exported_Functions */

/** @} */
/* End of group UART */

/** @} */
/* End of group TX04_Periph_Driver */

