/**
 ****************************************************************************
 * @file    ipdrv_enc.c
 * @brief   This file provides API functions for M4Kx A-ENC driver.
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#if (defined(__A_ENC) ||defined(__ENC))
/* Includes ------------------------------------------------------------------*/
#include "ipdrv_enc.h"

/** @addtogroup TX04_Periph_Driver
  * @{
  */

/** @defgroup ENC
  * @brief ENC driver modules
  * @{
  */

/** @defgroup ENC_Private_Defines
  * @{
  */

#define ENTNCR_ENRUN_SET               ((uint32_t)0x00000040)
#define ENTNCR_ENCLR_SET               ((uint32_t)0x00000400)
#define ENTNCR_SFTCAP_SET              ((uint32_t)0x00000800)
#define ENTNCR_CMPSEL_SET              ((uint32_t)0xEFFFFFFF)
#define ENTNCR_TRGCAP_SET              ((uint32_t)0xFFFFEFFF)

#define ENINPCR_NCT_MASK               ((uint32_t)0xFFFF80FF)
#define ENINPCR_SYNCSPLEN_MASK         ((uint32_t)0xFFFFFFFE)
#define ENINPCR_SYNCSPLMD_MASK         ((uint32_t)0xFFFFFFFD)
#define ENINPCR_SYNCNCZEN_MASK         ((uint32_t)0xFFFFFFFB)
#define ENINPCR_SYNCSPLEN_SET          ((uint32_t)0x00000001)
#define ENINPCR_SYNCSPLMD_SET          ((uint32_t)0x00000002)

#define ENTNCR_MODE_MASK               ((uint32_t)0x000E0000)


/** @} */
/* End of group ENC_Private_Defines */


/** @defgroup ENC_Private_FunctionPrototypes
  * @{
  */

/** @} */
/* End of group ENC_Private_FunctionPrototypes */

/** @defgroup ENC_Private_Functions
  * @{
  */

/** @} */
/* End of group group ENC_Private_Functions */

/** @defgroup ENC_Exported_Functions
  * @{
  */

/**
  * @brief  Enable the specified encoder operation.
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @retval None.
  * @register The used register:
  *   ENCTNCR<ENRUN>    
  */
void ENC_Enable(TSB_EN_TypeDef * ENx)
{
    uint32_t tmp = 0U;
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    tmp = ENx->TNCR;
    /* Set ENTNCR <ENRUN> to enable encoder operation. */
    tmp |= ENTNCR_ENRUN_SET;
    ENx->TNCR = tmp;
}

/**
  * @brief  Initialize the specified encoder channel.
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @param  InitStruct: The structure containing basic encoder configuration.
  * @retval None.
  * @register The used register:
  *   ENCTNCR<CMPSEL><UDMD[1:0]><TOVMD><MCMPMD><DECMD><SDTEN><MODE[2:0]><P3EN>
  *   <TRGCAPMD><ZESEL><ZEN><ENDEV[2:0]>
  */
void ENC_Init(TSB_EN_TypeDef * ENx, ENC_InitTypeDef * InitStruct)
{
    uint32_t tmp = 0U;
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    assert_param(IS_POINTER_NOT_NULL(InitStruct));
    assert_param(IS_ENC_MODE(InitStruct->ModeType));
    assert_param(IS_ENC_DIV_FACTOR(InitStruct->PulseDivFactor));
    /* Configure the encoder mode select  */
    if ((InitStruct->ModeType == ENC_ENCODER_MODE) || (InitStruct->ModeType == ENC_TIMER_MODE) ||
        (InitStruct->ModeType == ENC_PHASE_COUNTER_MODE)) {
        /* Check the parameters */
        assert_param(IS_ENC_ZPHASE_STATUS(InitStruct->ZphaseStatus));
        /*  Enable or disable z-phase detected */
        tmp = (InitStruct->ModeType << 17U) | (InitStruct->ZphaseStatus << 7U);
    } else {
        /* Check the parameters */
        assert_param(IS_ENC_PHASE(InitStruct->PhaseType));
        /* 2-phase / 3-phase input selection */
        tmp = (InitStruct->ModeType << 17U) | (InitStruct->PhaseType << 16U);
    }
    if ((InitStruct->ModeType == ENC_TIMER_MODE)
        || (InitStruct->ModeType == ENC_PHASE_COUNTER_MODE)) {
        /* Check the parameters */
        assert_param(IS_ENC_EDGE(InitStruct->EdgeType));
        /* Edge selection of ENCZ  */
        tmp |= (InitStruct->EdgeType << 8U);
    } else {
        /* Do nothing  */
    }
    if (InitStruct->ModeType == ENC_TIMER_MODE) {
        /* Check the parameters */
        assert_param(IS_ENC_CLEAR_CONDITION(InitStruct->SetClearCount));
        /* clear register for Counter */
        tmp |= (InitStruct->SetClearCount << 28U);
    } else {
        /* Do nothing  */
    }
    if ((InitStruct->ModeType == ENC_SENSOR_PHASE_MODE)
        || (InitStruct->ModeType == ENC_PHASE_COUNTER_MODE)) {

        /* Check the parameters */
        assert_param(IS_ENC_COUNT_MODE(InitStruct->UpDownType));
        /* set up or down counter RATE */
        tmp |= (InitStruct->UpDownType << 26U);
    } else {
        /* Do nothing  */
    }
    if ((InitStruct->ModeType == ENC_ENCODER_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_EVENT_MODE)) {
        /* Do nothing  */
    } else {
        /* Check the parameters */
        assert_param(IS_ENC_COUNT_STOP(InitStruct->CountOperation));
        /* set counter RELOAD operation */
        tmp |= (InitStruct->CountOperation << 25U);
    }
    if ((InitStruct->ModeType == ENC_SENSOR_TIME_MODE) || (InitStruct->ModeType == ENC_TIMER_MODE)) {
        /* Check the parameters */
        assert_param(IS_ENC_COMPARE_MODE(InitStruct->CompareMode));
        /* Compare  match or size */
        tmp |= (InitStruct->CompareMode << 24U);
    } else {
        /* Do nothing  */
    }
    if ((InitStruct->ModeType == ENC_ENCODER_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_EVENT_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_TIME_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_PHASE_MODE)) {

        /* Check the parameters */
        assert_param(IS_ENC_EDGE_DIRECTION(InitStruct->DetectionDirection));
        /* Sets the detection direction for the decoder */
        tmp |= (InitStruct->DetectionDirection << 22U);
    } else {
        /* Do nothing  */
    }
    if ((InitStruct->ModeType == ENC_SENSOR_TIME_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_EVENT_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_PHASE_MODE)) {
        /* Check the parameters */
        assert_param(IS_ENC_DETECT_SKIP(InitStruct->SkipDetection));
        /* Set skip detection at 3-phase decoding */
        tmp |= (InitStruct->SkipDetection << 21U);
    } else {
        /* Do nothing  */
    }
    if ((InitStruct->ModeType == ENC_TIMER_MODE) || (InitStruct->ModeType == ENC_PHASE_COUNTER_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_TIME_MODE)
        || (InitStruct->ModeType == ENC_SENSOR_PHASE_MODE)) {
        /* Check the parameters */
        assert_param(IS_ENC_CAPTURE_MODE(InitStruct->SelectCapture));
        /* Set Trigger capture operationselection */
        tmp |= (InitStruct->SelectCapture << 12U);
    } else {
        /* Do nothing  */
    }
    /* Configure the encoder input control register */
    tmp |= (InitStruct->PulseDivFactor);
    ENx->TNCR = tmp;
}

/**
  * @brief  Get operation mode.
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @retval getMode: Get the encoder operation mode.
  *   This parameter can be one of the following values:
  *   ENC_ENCODER_MODE, ENC_SENSOR_EVENT_MODE, ENC_SENSOR_TIME_MODE,
  *   ENC_TIMER_MODE, ENC_SENSOR_PHASE_MODE, ENC_PHASE_COUNTER_MODE.   
  * @register The used register:
  *   ENCTNCR<MODE[2:0]>
  */
uint32_t ENC_GetControlMode(TSB_EN_TypeDef * ENx)
{
    uint32_t getMode = 0U;
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    /* Read Operation Mode */
    getMode = ENx->TNCR;
    getMode &= ENTNCR_MODE_MASK;
    getMode >>= 17U;
    return getMode;
}

/**
  * @brief  Get status of flag
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @retval The encoder flag.
  * @register The used register:
  *   ENCSTS<REVERR><UD><ZDET><SKPDT><PDERR><INERR>
  */
ENC_FlagStatus ENC_GetENCFlag(TSB_EN_TypeDef * ENx)
{
    ENC_FlagStatus retval = { 0U };
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    /* get status */
    retval.All = ENx->STS;
    return retval;
}

/**
  * @brief  Set the encoder counter period
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @param  PeriodValue: Set the encoder counter period
  *   In sensor mode (timer count) and timer mode,
  *   This parameter can be 0x00000000 - 0xFFFFFFFF.
  *   In other mode,
  *   This parameter can be 0x0000 - 0xFFFF.
  * @retval None.
  * @register The used register:
  *   ENCRELOAD<RELOADH[15:0]><RELOADL[15:0]>
  */
void ENC_SetCounterReload(TSB_EN_TypeDef * ENx, uint32_t PeriodValue)
{
    uint32_t mode = 0U;
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    /* Get operation mode */
    mode = ENC_GetControlMode(ENx);
    /* Set the encoder counter period */
    if ((mode == ENC_SENSOR_TIME_MODE) || (mode == ENC_TIMER_MODE)) {
        ENx->RELOAD = PeriodValue;
    } else {
        assert_param(IS_ENC_16BITVALUE(PeriodValue));
        ENx->RELOAD = (PeriodValue << 16U);
    }
}

/**
  * @brief  Get the counter value
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @retval  value of the couter.
  * @register The used register:
  *   ENCCNT<CNTH[15:0]><CNTL[15:0]>
  */
uint32_t ENC_GetCounterValue(TSB_EN_TypeDef * ENx)
{
    uint32_t retval = 0U;
    uint32_t mode = 0U;
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    /* Get operation mode */
    mode = ENC_GetControlMode(ENx);
    /* Get the counter value */
    if ((mode == ENC_SENSOR_TIME_MODE) || (mode == ENC_TIMER_MODE)) {
        retval = ENx->CNT;
    } else {
        retval = ENx->CNT;
        retval = (retval >> 16U);
    }
    return retval;
}

/**
  * @brief  Set Noise Cancel Time
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @param  Time: noise cancel time.
  *   This parameter can be one of the following values:
  *   0x00U to 0x7FU. 
  * @retval None.
  * @register The used register:
  *   ENCINPCR<NCT[7:0]>
  */
void ENC_SetNoiseCancelTime(TSB_EN_TypeDef * ENx, uint32_t Time)
{
    uint32_t tmp = 0U;
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    assert_param(IS_ENC_7BITVALUE(Time));
    /* Set Noise Cancel Time */
    tmp = ENx->INPCR;
    tmp &= ENINPCR_NCT_MASK;
    tmp |= (Time << 8U);
    ENx->INPCR = tmp;
}

/**
  * @brief  Sets the sampling frequency 
  * @param  ENx: Select the encoder channel.
  *   This parameter can be one of the following values:
  *   EN0.
  * @param  SampleClk: Sample Clock
  *   This parameter can be one of the following values:
  *   ENC_SAMPLE_CLK_FSYS_1,ENC_SAMPLE_CLK_FSYS_2,ENC_SAMPLE_CLK_FSYS_4,
  *   ENC_SAMPLE_CLK_FSYS_8
  * @retval None.
  * @register The used register:
  *   ENCCLKCR<SPLCKS[1:0]>
  */
void ENC_SetSampleClk(TSB_EN_TypeDef * ENx, uint32_t SampleClk)
{
    /* Check the parameters */
    assert_param(IS_ENC_PERIPH(ENx));
    assert_param(IS_ENC_FSYS_FACTOR(SampleClk));
    /* set sample clock   */
    ENx->CLKCR = SampleClk;
}

/** @} */
/* End of group ENC_Exported_Functions */

/** @} */
/* End of group ENC */

/** @} */
/* End of group TX04_Periph_Driver */
#endif
