/**
 ****************************************************************************
 * @file	E_Sub.c
 * @brief	Sub Routine Functions
 * @version	V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#include "ipdefine.h"

#define DEFINE_APP
#include "E_Sub.h"
#undef DEFINE_APP


/**
 *********************************************************************************************
 * @brief		Cal Sine
 *				sin(x) = 3.140625x + 0.02026367x^2 -5.325196x^3 + 0.5446778x^4 + 1.800293x^5
 *
 * @param		short	theta	theta
 *
 * @return		short	sin(theta)
 *********************************************************************************************
 */
#define cSine1		(short)(FIX_12 *  3.140625)
#define cSine2		(short)(FIX_12 *  0.02026367)
#define cSine3		(short)(FIX_12 * -5.325196)
#define cSine4		(short)(FIX_12 *  0.5446778)
#define cSine5		(short)(FIX_12 *  1.800293)

short E_Sine(short theta)
{
	short x;
	short x1;
	long temp32;

	x = (short)(theta & 0x7fff);
	if (x >= ELE_DEG(90))
	{
		x = (q15_t)(ELE_DEG(180) - x);
	}
	x1 = x;
	temp32 = (cSine1 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine2 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine3 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine4 * (long)x1) << 1;
	x1 = (short)(x1 * (long)x >> (16 - 1));
	temp32 += (cSine5 * (long)x1) << 1;
	temp32 <<= 3;

	if ((unsigned long)temp32 >= 0x7fffffff)
	{
		temp32 = 0x7fffffff;
	}
	if (theta <= 0)
	{
		temp32 *= ( -1);
	}

	return ((short)(temp32 >> 16));
}


/**
 *********************************************************************************************
 * @brief		E_Cosine
 *
 * @param		short	theta
 *
 * @return		short
 *********************************************************************************************
 */
short E_Cosine(short theta)
{
	return (E_Sine((short)(ELE_DEG(90) + theta)));
}

/**
 *********************************************************************************************
 * @brief		E_Clarke
 *
 * @param		q15_t 		_iu			U-phase current
 * @param		q15_t 		_iv			V-phase current
 * @param		q15_t 		_iw			W-phase current
 * @param		q15_t* 		_ialpha		alpha-axis current storage address
 * @param		q15_t* 		_ibeta		beta-axis current storage address
 *
 * @return		none	
 *********************************************************************************************
 */
void E_Clarke(q15_t _iu, q15_t _iv, q15_t _iw, q15_t* _ialpha, q15_t* _ibeta)
{

	int32_t temp_1;

	/* Ialpha=Iu */
	*_ialpha = _iu;
	/* Ibeta=(Iu + 2*Iv)/sqrt3 */
	temp_1 = ((int32_t)_iu  + (int32_t)(_iv * 2));
	*_ibeta = (temp_1 * (int32_t)((1L << 15) * (1.0f / 1.7320508f))) >> 15;
}

/**
 *********************************************************************************************
 * @brief		E_Park
 *
 * @param		q15_t 		_ialpha		alpha-axis 
 * @param		q15_t		_ibeta		bata-axis
 * @param		uint16_t	theta		Theta(position)
 * @param		q15_t* 		_id			d-axis storage address
 * @param		q15_t* 		_iq			q-axis storage address
 *
 * @return		none	
 *********************************************************************************************
 */
void E_Park(q15_t _ialpha, q15_t _ibeta, uint16_t _theta, q15_t* _id, q15_t* _iq)
{
	int32_t temp_1, temp_2, temp_3, temp_4;
	int16_t temp_sin, temp_cos;

	/*alpha_beta/dq conversion (Park.t) */
	temp_cos = E_Cosine(_theta);
	temp_sin = E_Sine(_theta);

	/* Id = Ialpha * cos(theta)+Ibeta * sin(theta) */
	temp_1 = ((int32_t)_ialpha * temp_cos) << 1;
	temp_2 = ((int32_t)_ibeta  * temp_sin) << 1;
	*_id = (q15_t)(__QADD(temp_1, temp_2) >> 16);

	/* Iq = Ialpha * -sin(theta) + Ibeta * cos(theta) */
	temp_3 = ((int32_t)_ialpha * (int32_t)(-temp_sin)) << 1;
	temp_4 = ((int32_t)_ibeta  * (int32_t)temp_cos) << 1;
	*_iq = (q15_t)(__QADD(temp_3, temp_4) >> 16);
}

/**
 *********************************************************************************************
 * @brief		E_InvPark
 *
 * @param		q31_t 		_d			d-axis
 * @param		q31_t 		_q			q-axis
 * @param		uint16_t	theta		Theta(position)
 * @param		q31_t* 		_alpha		alpha-axis storage address
 * @param		q31_t* 		_beta		bata-axis storage address
 *
 * @return		none	
 *********************************************************************************************
 */
void E_InvPark(q31_t _d, q31_t _q, uint16_t _theta, q31_t* _alpha, q31_t* _beta)
{
	int32_t temp_1, temp_2, temp_3, temp_4;
	int16_t temp_sin, temp_cos;

	temp_cos = E_Cosine(_theta);
	temp_sin = E_Sine(_theta);

	/* Valpha = Vd * cos(theta) - Vq * sin(theta) */
	temp_1 = (int32_t)(((int64_t)(_d) * (temp_cos)) >> 15);
	temp_2 = (int32_t)(((int64_t)(_q) * (temp_sin)) >> 15);
	*_alpha = __QSUB(temp_1, temp_2);

	/* Vbeta = Vd * sin(theta) + Vq * cos(theta) */
	temp_3 = (int32_t)(((int64_t)(_d) * (temp_sin)) >> 15);
	temp_4 = (int32_t)(((int64_t)(_q) * (temp_cos)) >> 15);
	*_beta = __QADD(temp_3, temp_4);
}

/**
 *********************************************************************************************
 * @brief		Get the maximum value of three data.
 *
 * @param		int32_t _a			Data a.
 * @param		int32_t _b			Data b.
 * @param		int32_t _c			Data c.
 *
 * @return		Max value
 *********************************************************************************************
 */
int32_t E_Max3(int32_t _a, int32_t _b, int32_t _c)
{
	int32_t	ret;

	if (_a >= _b)
	{
		ret = (_a < _c) ? _c : _a;
	}
	else
	{
		ret = (_b < _c) ? _c : _b;
	}
	return (ret);
}

/**
 *********************************************************************************************
 * @brief		Get the medium value of three data.
 *
 * @param		int32_t _a			Data a.
 * @param		int32_t _b			Data b.
 * @param		int32_t _c			Data c.
 *
 * @return		Medium value
 *********************************************************************************************
 */
int32_t E_Med3(int32_t _a, int32_t _b, int32_t _c)
{
	int32_t	ret;

	if (_a >= _b)
	{
		ret = (_c < _b) ? _b : (_c > _a) ? _a : _c;
	}
	else
	{
		ret = (_c < _a) ? _a : (_c > _b) ? _b : _c;
	}
	return (ret);
}

/**
 *********************************************************************************************
 * @brief		Get the minimum value of three data.
 *
 * @param		int32_t _a			Data a.
 * @param		int32_t _b			Data b.
 * @param		int32_t _c			Data c.
 *
 * @return		Minimum value
 *********************************************************************************************
 */
int32_t E_Min3(int32_t _a, int32_t _b, int32_t _c)
{
	int32_t	ret;

	if (_a >= _b)
	{
		ret = (_c < _b) ? _c : _b;
	}
	else
	{
		ret = (_c < _a) ? _c : _a;
	}
	return (ret);
}

/**
 ********************************************************************************************
 * @brief		E_Sort3.
 *
 * @param[in,out]	int32_t*	_n1	.
 * @param[in,out]	int32_t*	_n2	.
 * @param[in,out]	int32_t*	_n3	.
 *
 * @return		none
 ********************************************************************************************
 */
void E_Sort3(int32_t* _n1, int32_t* _n2, int32_t* _n3)
{
	int32_t	tmp;

	if (*_n1 > *_n2)
	{
		tmp = *_n1;
		*_n1 = *_n2;
		*_n2 = tmp;
	}
	if (*_n2 > *_n3)
	{
		tmp = *_n2;
		*_n2 = *_n3;
		*_n3 = tmp;
	}

	if (*_n1 > *_n2)
	{
		tmp = *_n1;
		*_n1 = *_n2;
		*_n2 = tmp;
	}
}

/****************************************************************************************
* Module	:   E_Sqrt
*---------------------------------------------------------------------------------------
* Function  :   Sqrt calculation
* Input	 :   data_s
* Output	:   sqrt(data_s)
* Note	  :   none
* Ver	   :   1.000
****************************************************************************************/
const unsigned short Table_Sqrt[129] =
{
	0, 2896, 4096, 5017, 5793, 6476, 7094, 7663, 8192, 8689, 9159, 9606, 10033, 10443, 10837, 11217, 11585, 11942, 12288, 12625,
	12953, 13273, 13585, 13890, 14189, 14482, 14768, 15050, 15326, 15597, 15864, 16126, 16384, 16638, 16888, 17135, 17378,
	17618, 17854, 18087, 18318, 18545, 18770, 18992, 19212, 19429, 19644, 19856, 20066, 20274, 20480, 20684, 20886, 21085,
	21283, 21480, 21674, 21867, 22058, 22247, 22435, 22621, 22806, 22989, 23170, 23351, 23530, 23707, 23884, 24059, 24232,
	24405, 24576, 24746, 24915, 25083, 25249, 25415, 25580, 25743, 25905, 26067, 26227, 26387, 26545, 26703, 26859, 27015,
	27170, 27324, 27477, 27629, 27780, 27931, 28081, 28230, 28378, 28525, 28672, 28818, 28963, 29108, 29251, 29394, 29537,
	29678, 29819, 29960, 30099, 30238, 30377, 30515, 30652, 30788, 30924, 31059, 31194, 31328, 31462, 31595, 31727, 31859,
	31991, 32122, 32252, 32382, 32511, 32640, 32768
};

q31_t C_E_Sqrt(int32_t data_s)
{
	int32_t temp1 = 0, temp2 = 0;
	int32_t acc1 = 0, acc41 = 0, acc42 = 0, acc43 = 0;
	q31_t data_result;

	temp1 = data_s;
	temp2 = 1;

	if (data_s < (0x80000000 / 4))
	{
		temp1 = data_s * 4;
		temp2 = 2;
	}

	if (data_s < (0x80000000 / 16))
	{
		temp1 = data_s * 16;
		temp2 = 4;
	}

	if (data_s < (0x80000000 / 64))
	{
		temp1 = data_s * 64;
		temp2 = 8;
	}

	if (data_s < (0x80000000 / 256))
	{
		temp1 = data_s * 256;
		temp2 = 16;
	}

	if (data_s < (0x80000000 / 1024))
	{
		temp1 = data_s * 1024;
		temp2 = 32;
	}

	if (data_s < (0x80000000 / 4096))
	{
		temp1 = data_s * 4096;
		temp2 = 64;
	}

	acc1 = (temp1 >> (32 - 7 - 1));
	acc43 = temp1 % (0x80000000 / (1 << 7));
	acc41 = Table_Sqrt[acc1];
	acc42 = Table_Sqrt[acc1 + 1];
	data_result = ((acc41 + ((acc42 - acc41) * acc43) / (0x80000000 / (1 << 7))) / temp2) << 16;
	return (data_result);
}

/****************************************************************************************
* Module   :   muljc
*---------------------------------------------------------------------------------------
* Function  :   multiplication
* Input     :   r1,c1
* Output    :   r1*c1
* Note      :   none
* Ver       :   1.000
****************************************************************************************/

q31_t C_muljc(q31_t r1, q31_t c1)
{
	q31_t tempm = 0, n = 0, tempn, tempp;

	n = 0;

	while ((c1 >> n) >= (256 * 128))
	{
		n = n + 1;
	}

	tempm = r1 * (c1 >> n);
	tempn = cCAL_FACTOR >> (n + 1);
	tempp = (((-1) * 256 * 256 * 256 * 128) >> (n + 1));
	if (tempm >= tempn)
	{
		tempm = tempn;
	}

	if (tempm < tempp)
	{
		tempm = tempp;
	}

	return tempm << (n + 1);
}

/**
 *********************************************************************************************
 * @brief       Limitation.
 *
 * @param       int64_t         _indata         Q31 input data(Double precision).
 * @param       int32_t         _limlo          Q31 lower limit value.
 * @param       int32_t         _limup          Q31 upper limit value.
 *
 * @return      int32_t         Q31
 *********************************************************************************************
 */
int32_t Limit_sub(int64_t _indata, int32_t _limlo, int32_t _limup)
{
    int32_t ret = (int32_t)_indata;

    if (_indata > (int64_t)_limup)
    {
        ret = _limup;
    }
    if (_indata < (int64_t)_limlo)
    {
        ret = _limlo;
    }
    return (ret);
}

/**************************************************************************/

/*note:

this caculation is that r1 * c1
r1---VAR32
c1---VAR32
r1max = 32768
c1max = 32768
so, c1 !> 32768
so,c1 need right move n bit that  cann't make r1*c1 overflow.
and, the result is r1*c1 << (n+1)(+1 means that (signed short * signed short <<1 = signed long))*/

/**************************************************************************/

q31_t muljc(q31_t r1, q31_t c1)
{
	q31_t tempm = 0, n = 0;

	n = 0;
	while ((c1 >> n) >= (256 * 128))
	{
		n = n + 1;
	}

	tempm = r1 * (c1 >> n);

	if (tempm >= cCAL_FACTOR >> (n + 1))	/* make data not overflow */
	{
		tempm = cCAL_FACTOR >> (n + 1);
	}
	if (tempm < (((-1) * 256 * 256 * 256 * 128) >> (n + 1)))	/* make data not overflow */
	{
		tempm = (((-1) * 256 * 256 * 256 * 128) >> (n + 1));
	}

	return tempm << (n + 1);
}

/****************************************************************************************
* Module	: smul32
*---------------------------------------------------------------------------------------
* Function : A*B
* Input	 : a,b
* Output   : -
* Work	 : -
* Note	 : -
****************************************************************************************/
q31_t smul32(q31_t a, q31_t b)
{
	q31_t c = 0;
	if (a < 0)
	{
		a = 0x7fffffff;
	}
	c = (a * b) << 1;
	if (b > 0 && c < 0)
	{
		c = 0x7fffffff;
	}
	else if (b < 0 && c > 0)
	{
		c = 0x80000000;
	}
	return c;
}

/**
 *********************************************************************************************
 * @brief	   U_command_limit_sub function.
 *
 * @param	   uint32_t	_now					Now value
 * @param	   uint32_t	_target				 Target value
 * @param	   uint32_t	_lim_up				 Acceleration value
 * @param	   uint32_t	_lim_down			   Deceleration value
 *
 * @return	  uint32_t							Update value
 *********************************************************************************************
 */
uint32_t U_C_command_limit_sub(uint32_t _now, uint32_t _target, uint32_t _lim_up, uint32_t _lim_down)
{
	uint32_t  lim_target;

	if (_now == _target)
	{
		lim_target = _target;
	}
	else
	{
		{
			/* now speed is positive */
			if (_now < _target)
			{
				/* UP */
				if ((0xFFFFFFFF - _lim_up) > _now)
				{
					lim_target = _now + _lim_up;
				}
				else
				{
					lim_target = 0xFFFFFFFF;
				}
				lim_target = ((_target > lim_target)? lim_target : _target);
			}
			else
			{
				/* DOWN */
				if (_now > _lim_down)
				{
					lim_target = _now - _lim_down;
					lim_target = ((_target < lim_target)? lim_target : _target);
				}
				else
				{
					lim_target = _target;
				}
			}
		}
	}
	return lim_target;
}
/*********************************** END OF FILE ******************************/
