/**
 ****************************************************************************
 * @file	B_User.c
 * @brief	Motor control for user Source File
 * @version V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2023
 *****************************************************************************
 */
#include <stdlib.h>

#include "ipdefine.h"
#include "D_Para.h"
#include "mcuip_drv.h"
#include "usercon.h"
#include "HPFC_drv.h"
#include "HPFC_para.h"
#include "D_Driver.h"
#include "C_Control.h"

#define DEFINE_APP
#include "B_User.h"
#undef  DEFINE_APP


/*===================================================================*
	  Local Variable Definition
 *===================================================================*/

/*===================================================================*
	Function
 *===================================================================*/
/**
 *********************************************************************************************
 * @brief		Initialize at Motor control
 *
 * @return		none
 *********************************************************************************************
 */
void	B_Motor_Init(void)
{
	error_t* _error = &Error;
#if defined(__CONTROL_MOTOR_COMP)

	/***** Setting Motor type  *****/
	Motor_comp.shunt_type = (shunt_type_e)cSHUNT_TYPE_COMP;
	Motor_comp.boot_type = cBOOT_TYPE_COMP;
	Motor_comp.motor_type = cCOMP;
	Motor_comp.para.V_max = cV_MAX_COMP;
	Motor_comp.para.I_max = cA_MAX_COMP * 10;
	Motor_comp.para.Hz_max = cHZ_MAX_COMP;
	Motor_comp.spd_power_limit = cHZ_E_NORMAL_COMP;
#if defined(__USE_ENCODER_COMP)
	/* Encoder ON:1 OFF:0 */
	Motor_comp.usr.com_user.encoder = CLEAR;
#endif /* __USE_ENCODER_COMP */

	/***** Initialize variable *****/
	Motor_comp.stage.main = cStop;
	Motor_comp.stage.sub = cStep0;

	Motor_comp.drv.Iao_ave.word = Motor_comp.drv.Ibo_ave.word = Motor_comp.drv.Ico_ave.word = ADC_V2HEX(cSHUNT_ZERO_OFFSET_COMP) << 16;
#if (cSHUNT_TYPE_COMP == 1)
	Motor_comp.drv.Idco_ave.word = ADC_V2HEX(cSHUNT_ZERO_OFFSET_COMP) << 16;
#endif

	/***** user data setting *****/
	/* Start Current (Iq) command */
	Motor_comp.usr.Iq_st_user = (q15_t)(cIQ_ST_USER_ACT_COMP * FIX_15 / cA_MAX_COMP);

	/* Start Current (Id) command */
	Motor_comp.usr.Id_st_user = (q15_t)(cID_ST_USER_ACT_COMP * FIX_15 / cA_MAX_COMP);

	/* Set initial position */
	Motor_comp.usr.lambda_user = ELE_DEG(cINITIAL_POSITION_COMP);

	/* Set Phase 3phase:0 2phase:1 */
	Motor_comp.usr.com_user.modul = c2PHASE;

	/* Set Shift-PWM ON:1 OFF:0 for 1shunt */
	Motor_comp.usr.com_user.spwm = 1;
	/***** Set parameter *****/
	/* Motor parameter */
	Motor_comp.para.motor.r = (q31_t)(FIX_15 * cMOTOR_R_COMP * cA_MAX_COMP / cV_MAX_COMP);
	Motor_comp.para.motor.Lq = (q31_t)(FIX_15 * PAI2 * cA_MAX_COMP * cHZ_MAX_COMP * cMOTOR_LQ_COMP / 1000 / cV_MAX_COMP);
	Motor_comp.para.motor.Ld = (q31_t)(FIX_15 * PAI2 * cA_MAX_COMP * cHZ_MAX_COMP * cMOTOR_LD_COMP / 1000 / cV_MAX_COMP);
	/* Set Shift-PWM threshold for 1shunt */
	Motor_comp.para.spwm_threshold = (q15_t)(cHZ_SPWM_COMP * FIX_15 / cHZ_MAX_COMP);

	/* Set check pulse width */
#if defined(__USE_VE_COMP)
 #if (cSHUNT_TYPE_COMP == 1)
	/* for 1shunt VE */
	Motor_comp.para.chkpls = (uint16_t)(FIX_15 * cMINPLS_COMP / cPWMPRD_COMP) * 2;
 #else /* cSHUNT_TYPE_COMP == 1 */
	/* for 3shunt VE */
	/* Convert On width to Off width for VE Register. */
	Motor_comp.para.chkpls = (uint16_t)(FIX_15 - (FIX_15 * cMAXPLS_COMP / cPWMPRD_COMP));
 #endif	/* cSHUNT_TYPE_COMP == 1 */
#else
 #if (cSHUNT_TYPE_COMP == 1)
	/* for 1shunt soft */
	Motor_comp.para.chkpls = (uint16_t)(FIX_15 * cMINPLS_COMP / cPWMPRD_COMP);
 #else /* cSHUNT_TYPE_COMP == 1 */
	/* for 3shunt soft */
	/* On width */
	Motor_comp.para.chkpls = (uint16_t)(FIX_15 * cMAXPLS_COMP / cPWMPRD_COMP);
 #endif /* cSHUNT_TYPE_COMP == 1 */
#endif /* defined(__USE_VE_COMP) */

	Motor_comp.para.vd_pos = (q31_t)(FIX_31 * cVD_POS_COMP / cV_MAX_COMP);
	Motor_comp.para.spd_coef = (q31_t)(FIX_15 * cSPD_COEF_COMP);

	/* Acceleration setting */
	Motor_comp.para.sp_ud_lim_f.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cFCD_UD_LIM_COMP / cHZ_MAX_COMP);  /* Speed up/down limit at Force */
	Motor_comp.para.sp_up_lim_s.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cSTD_UP_LIM_COMP / cHZ_MAX_COMP);  /* Speed up limit at Steady */
	Motor_comp.para.sp_dn_lim_s.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cSTD_DW_LIM_COMP / cHZ_MAX_COMP);  /* Speed down limit at Steady */

	/* Time setting */
	Motor_comp.para.time.bootstp = (uint16_t)(cBOOT_LEN_COMP / cMAINLOOP_PRD); 							/* Time of Bootstrap */
	Motor_comp.para.time.initpos = (uint16_t)(cINIT_LEN_COMP / cMAINLOOP_PRD);							/* Time of Positioning */
	Motor_comp.para.time.initpos2 = (uint16_t)(cINIT_WAIT_LEN_COMP / cMAINLOOP_PRD);					/* Time of Positioning wait */
	Motor_comp.para.time.go_up = (uint16_t)(cGOUP_DELAY_LEN_COMP / cMAINLOOP_PRD);						/* Time of Change-up */

	/* Omega setting */
	Motor_comp.para.omega_min = (q15_t)(FIX_15 * cHZ_MIN_COMP / cHZ_MAX_COMP);
	Motor_comp.para.omega_v2i = (q15_t)(FIX_15 * cHZ_V2I_COMP / cHZ_MAX_COMP);
	Motor_comp.para.delta_lambda = ((q31_t)(FIX_15 * cHZ_MIN_COMP / cHZ_MAX_COMP) * (q31_t)(FIX_16 * cHZ_MAX_COMP * cMAINLOOP_PRD)) << 1;

	/* Current con parameter */
#if defined(__CURRENT_DKI_RANGE)
	Motor_comp.para.crt.dki = (q31_t)(FIX_12 * cID_KI_COMP * cCTRL_PRD_COMP * cA_MAX_COMP / cV_MAX_COMP);
#else
	Motor_comp.para.crt.dki = (q31_t)(FIX_15 * cID_KI_COMP * cCTRL_PRD_COMP * cA_MAX_COMP / cV_MAX_COMP);
#endif
#if defined(__CURRENT_DKP_RANGE)
	Motor_comp.para.crt.dkp = (q31_t)(FIX_12 * cID_KP_COMP * cA_MAX_COMP / cV_MAX_COMP);
#else
	Motor_comp.para.crt.dkp = (q31_t)(FIX_15 * cID_KP_COMP * cA_MAX_COMP / cV_MAX_COMP);
#endif
#if defined(__CURRENT_QKI_RANGE)
	Motor_comp.para.crt.qki = (q31_t)(FIX_12 * cIQ_KI_COMP * cCTRL_PRD_COMP * cA_MAX_COMP / cV_MAX_COMP);
#else
	Motor_comp.para.crt.qki = (q31_t)(FIX_15 * cIQ_KI_COMP * cCTRL_PRD_COMP * cA_MAX_COMP / cV_MAX_COMP);
#endif
#if defined(__CURRENT_QKP_RANGE)
	Motor_comp.para.crt.qkp = (q31_t)(FIX_12 * cIQ_KP_COMP * cA_MAX_COMP / cV_MAX_COMP);
#else
	Motor_comp.para.crt.qkp = (q31_t)(FIX_15 * cIQ_KP_COMP * cA_MAX_COMP / cV_MAX_COMP);
#endif

	/* Position est parameter */
#if defined(__POSITION_KI_RANGE)
	Motor_comp.para.pos.ki = (q31_t)(FIX_12 * cPOSITION_KI_COMP * cCTRL_PRD_COMP * cV_MAX_COMP / cHZ_MAX_COMP);
#else
	Motor_comp.para.pos.ki = (q31_t)(FIX_15 * cPOSITION_KI_COMP * cCTRL_PRD_COMP * cV_MAX_COMP / cHZ_MAX_COMP);
#endif

#if defined(__POSITION_KP_RANGE)
	Motor_comp.para.pos.kp = (q31_t)(FIX_12 * cPOSITION_KP_COMP * cV_MAX_COMP / cHZ_MAX_COMP);
#else
	Motor_comp.para.pos.kp = (q31_t)(FIX_15 * cPOSITION_KP_COMP * cV_MAX_COMP / cHZ_MAX_COMP);
#endif
	Motor_comp.para.pos.ctrlprd = (uint32_t)(FIX_16 * cHZ_MAX_COMP * cCTRL_PRD_COMP);
	Motor_comp.para.ctrlprd = cPWMPRD_COMP;

	/* Speed con parameter */
#if defined(__SPEED_KI_RANGE)
	Motor_comp.para.spd.ki = (q31_t)(FIX_12 * cSPEED_KI_COMP * cSPD_CTRL_PRD_COMP * cHZ_MAX_COMP / cA_MAX_COMP);
#else
	Motor_comp.para.spd.ki = (q31_t)(FIX_15 * cSPEED_KI_COMP * cSPD_CTRL_PRD_COMP * cHZ_MAX_COMP / cA_MAX_COMP);
#endif

#if defined(__SPEED_KP_RANGE)
	Motor_comp.para.spd.kp = (q31_t)(FIX_12 * cSPEED_KP_COMP * cHZ_MAX_COMP / cA_MAX_COMP);
#else
	Motor_comp.para.spd.kp = (q31_t)(FIX_15 * cSPEED_KP_COMP * cHZ_MAX_COMP / cA_MAX_COMP);
#endif

	/* current limit */
	Motor_comp.para.iq_lim = (q31_t)(FIX_31 * cIQ_LIM_COMP / cA_MAX_COMP);
	Motor_comp.para.id_lim = (q31_t)(FIX_31 * cID_LIM_COMP / cA_MAX_COMP);
	Motor_comp.para.err_ovc = (q15_t)(FIX_15 * cOVC_COMP / cA_MAX_COMP);

	/* Encoder parameter */
#if defined(__USE_ENCODER_COMP)
	Motor_comp.para.enc.pls2theta = ((FIX_32 / (cENC_PULSE_NUM_COMP * cENCMULTIPLE)) * cPOLEPAIR_COMP);
	Motor_comp.para.enc.deg_adjust = ELE_DEG32(cENC_DEG_ADJUST_COMP);
	Motor_comp.para.enc.plsnum = (cENC_PULSE_NUM_COMP * cENCMULTIPLE);
	Motor_comp.para.enc.pls2omega = (q15_t)(FIX_15 / ((cENC_PULSE_NUM_COMP * cENCMULTIPLE) / cPOLEPAIR_COMP) / cCTRL_PRD_COMP / cHZ_MAX_COMP);
	Motor_comp.para.enc.ctrlprd = (uint32_t)(FIX_16 * cHZ_MAX_COMP * cCTRL_PRD_COMP);
#endif /* __USE_ENCODER_COMP */

#if !defined(__USE_VE_COMP)
	Motor_comp.para.PhCvMd = cPHCVMD_SVM;
#if (cSHUNT_TYPE_COMP == 1)
	Motor_comp.para.TrgPosMd = cTRGPOS_1SHUNT_SECOND_HALF;
#else
	Motor_comp.para.TrgPosMd = cTRGPOS_3SHUNT;
#endif

	/* Trigger compensation value setting. (Trigger generate sawtooth wave.) */
	Motor_comp.para.TrgCompen = (int16_t)(FIX_15 * cTRG_COMPEN_COMP / (cPWMPRD_COMP / 2));

#endif /* !defined(__USE_VE_COMP) */

	/*Power caculation parameter */
	Motor_comp.power.power_cnt = 0;
	Motor_comp.power.power_para = cPOWER_FILTER;
	Motor_comp.constant.constant_filter1 = cFILTERCONSTANT1;
	Motor_comp.constant.constant_filter2 = cFILTERCONSTANT2;
	Motor_comp.constant.constant_filter3 = cFILTERCONSTANT3;
	Motor_comp.constant.constant_filter5 = cFILTERCONSTANT5;
	Motor_comp.filter_shift1 = cSHIFT1;
	Motor_comp.filter_shift2 = cSHIFT2;
	Motor_comp.Iac_eff_filter_shift3 = cSHIFT3;
	Motor_comp.Vac_eff_filter_shift4 = cSHIFT4;
	Motor_comp.power.pfc_max_current = (uint16_t)(cA_MAX_PFC * 10);
	Motor_comp.power.Vac_max = cVAC_MAX;

#endif /* __CONTROL_MOTOR_COMP */

#if defined(__CONTROL_MOTOR_FAN)
	/***** Data Initialize for FAN ********************************************/

	/***** Setting Motor type  *****/
	Motor_fan.shunt_type = (shunt_type_e)cSHUNT_TYPE_FAN;
	Motor_fan.boot_type = cBOOT_TYPE_FAN;
	Motor_fan.motor_type = cFAN;
	Motor_fan.para.V_max = cV_MAX_FAN;
	Motor_fan.para.I_max = cA_MAX_FAN * 10;
	Motor_fan.para.Hz_max = cHZ_MAX_FAN;

#if defined(__USE_ENCODER_FAN)
	/* Encoder ON:1 OFF:0 */
	Motor_fan.usr.com_user.encoder = SET;
#endif /* __USE_ENCODER_FAN */

	/***** Initialize variable *****/
	Motor_fan.stage.main = cStop;
	Motor_fan.stage.sub = cStep0;

	Motor_fan.drv.Iao_ave.word = Motor_fan.drv.Ibo_ave.word = Motor_fan.drv.Ico_ave.word = ADC_V2HEX(cSHUNT_ZERO_OFFSET_FAN) << 16;
#if (cSHUNT_TYPE_FAN == 1)
	Motor_fan.drv.Idco_ave.word = ADC_V2HEX(cSHUNT_ZERO_OFFSET_FAN) << 16;
#endif

	/***** user data setting *****/
	/* Start Current (Iq) command */
	Motor_fan.usr.Iq_st_user = (q15_t)(cIQ_ST_USER_ACT_FAN * FIX_15 / cA_MAX_FAN);

	/* Start Current (Id) command */
	Motor_fan.usr.Id_st_user = (q15_t)(cID_ST_USER_ACT_FAN * FIX_15 / cA_MAX_FAN);

	/* Set initial position */
	Motor_fan.usr.lambda_user = ELE_DEG(cINITIAL_POSITION_FAN);

	/* Set Phase 3phase:0 2phase:1 */
	Motor_fan.usr.com_user.modul = c2PHASE;

	/* Set Shift-PWM ON:1 OFF:0 for 1shunt */
	Motor_fan.usr.com_user.spwm = 1;

	/***** Set parameter *****/
	/* Motor parameter */
	Motor_fan.para.motor.r = (q31_t)(FIX_15 * cMOTOR_R_FAN * cA_MAX_FAN / cV_MAX_FAN);
	Motor_fan.para.motor.Lq = (q31_t)(FIX_15 * PAI2 * cA_MAX_FAN * cHZ_MAX_FAN * cMOTOR_LQ_FAN / 1000 / cV_MAX_FAN);
	Motor_fan.para.motor.Ld = (q31_t)(FIX_15 * PAI2 * cA_MAX_FAN * cHZ_MAX_FAN * cMOTOR_LD_FAN / 1000 / cV_MAX_FAN);

	/* Set Shift-PWM threshold for 1shunt */
	Motor_fan.para.spwm_threshold = (q15_t)(cHZ_SPWM_FAN * FIX_15 / cHZ_MAX_FAN);

	/* Set check pulse width */
#if defined(__USE_VE_FAN)
  #if (cSHUNT_TYPE_FAN == 1)
	/* for 1shunt VE */
	Motor_fan.para.chkpls = (uint16_t)(FIX_15 * cMINPLS_FAN / cPWMPRD_FAN) * 2;
  #else /* cSHUNT_TYPE_FAN == 1 */
	/* for 3shunt VE */
	/* Convert On width to Off width for VE Register. */
	Motor_fan.para.chkpls = (uint16_t)(FIX_15 - (FIX_15 * cMAXPLS_FAN / cPWMPRD_FAN));
  #endif /* cSHUNT_TYPE_FAN == 1 */
#else /* defined(__USE_VE_FAN) */
  #if (cSHUNT_TYPE_FAN == 1)
	/* for 1shunt soft */
	Motor_fan.para.chkpls = (uint16_t)(FIX_15 * cMINPLS_FAN / cPWMPRD_FAN);

  #else	/* cSHUNT_TYPE_FAN == 1 */
	/* for 3shunt soft */
	/* On width */
	Motor_fan.para.chkpls = (uint16_t)(FIX_15 * cMAXPLS_FAN / cPWMPRD_FAN);
  #endif /* cSHUNT_TYPE_FAN == 1 */
#endif /* defined(__USE_VE_FAN) */

	Motor_fan.para.vd_pos = (q31_t)(FIX_31 * cVD_POS_FAN / cV_MAX_FAN);
	Motor_fan.para.spd_coef = (q31_t)(FIX_15 * cSPD_COEF_FAN);

	/* Acceleration setting */
	Motor_fan.para.sp_ud_lim_f.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cFCD_UD_LIM_FAN / cHZ_MAX_FAN);	/* Speed up/down limit at Force */
	Motor_fan.para.sp_up_lim_s.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cSTD_UP_LIM_FAN / cHZ_MAX_FAN);	/* Speed up limit at Steady */
	Motor_fan.para.sp_dn_lim_s.word = (q31_t)(FIX_31 * cMAINLOOP_PRD * cSTD_DW_LIM_FAN / cHZ_MAX_FAN);	/* Speed down limit at Steady */

	/* Time setting */
	Motor_fan.para.time.bootstp = (uint16_t)(cBOOT_LEN_FAN / cMAINLOOP_PRD);							/* Time of Bootstrap */
	Motor_fan.para.time.initpos = (uint16_t)(cINIT_LEN_FAN / cMAINLOOP_PRD);							/* Time of Positioning */
	Motor_fan.para.time.initpos2 = (uint16_t)(cINIT_WAIT_LEN_FAN / cMAINLOOP_PRD);						/* Time of Positioning wait */
	Motor_fan.para.time.go_up = (uint16_t)(cGOUP_DELAY_LEN_FAN / cMAINLOOP_PRD);						/* Time of Change-up */

	/* Omega setting */
	Motor_fan.para.omega_min = (q15_t)(FIX_15 * cHZ_MIN_FAN / cHZ_MAX_FAN);
	Motor_fan.para.omega_v2i = (q15_t)(FIX_15 * cHZ_V2I_FAN / cHZ_MAX_FAN);
	Motor_fan.para.delta_lambda = ((q31_t)(FIX_15 * cHZ_MIN_FAN / cHZ_MAX_FAN) * (q31_t)(FIX_16 * cHZ_MAX_FAN * cMAINLOOP_PRD)) << 1;

	/* Current con parameter */
#if defined(__CURRENT_DKI_RANGE)
	Motor_fan.para.crt.dki = (q31_t)(FIX_12 * cID_KI_FAN * cCTRL_PRD_FAN * cA_MAX_FAN / cV_MAX_FAN);
#else
	Motor_fan.para.crt.dki = (q31_t)(FIX_15 * cID_KI_FAN * cCTRL_PRD_FAN * cA_MAX_FAN / cV_MAX_FAN);
#endif
#if defined(__CURRENT_DKP_RANGE)
	Motor_fan.para.crt.dkp = (q31_t)(FIX_12 * cID_KP_FAN * cA_MAX_FAN / cV_MAX_FAN);
#else
	Motor_fan.para.crt.dkp = (q31_t)(FIX_15 * cID_KP_FAN * cA_MAX_FAN / cV_MAX_FAN);
#endif
#if defined(__CURRENT_QKI_RANGE)
	Motor_fan.para.crt.qki = (q31_t)(FIX_12 * cIQ_KI_FAN * cCTRL_PRD_FAN * cA_MAX_FAN / cV_MAX_FAN);
#else
	Motor_fan.para.crt.qki = (q31_t)(FIX_15 * cIQ_KI_FAN * cCTRL_PRD_FAN * cA_MAX_FAN / cV_MAX_FAN);
#endif
#if defined(__CURRENT_QKP_RANGE)
	Motor_fan.para.crt.qkp = (q31_t)(FIX_12 * cIQ_KP_FAN * cA_MAX_FAN / cV_MAX_FAN);
#else
	Motor_fan.para.crt.qkp = (q31_t)(FIX_15 * cIQ_KP_FAN * cA_MAX_FAN / cV_MAX_FAN);
#endif

	/* Position est parameter */
#if defined(__POSITION_KI_RANGE)
	Motor_fan.para.pos.ki = (q31_t)(FIX_12 * cPOSITION_KI_FAN * cCTRL_PRD_FAN * cV_MAX_FAN / cHZ_MAX_FAN);
#else
	Motor_fan.para.pos.ki = (q31_t)(FIX_15 * cPOSITION_KI_FAN * cCTRL_PRD_FAN * cV_MAX_FAN / cHZ_MAX_FAN);
#endif

#if defined(__POSITION_KP_RANGE)
	Motor_fan.para.pos.kp = (q31_t)(FIX_12 * cPOSITION_KP_FAN * cV_MAX_FAN / cHZ_MAX_FAN);
#else
	Motor_fan.para.pos.kp = (q31_t)(FIX_15 * cPOSITION_KP_FAN * cV_MAX_FAN / cHZ_MAX_FAN);
#endif
	Motor_fan.para.pos.ctrlprd = (uint32_t)(FIX_16 * cHZ_MAX_FAN * cCTRL_PRD_FAN);
	Motor_fan.para.ctrlprd = cPWMPRD_FAN;

	/* Speed con parameter */
#if defined(__SPEED_KI_RANGE)
	Motor_fan.para.spd.ki = (q31_t)(FIX_12 * cSPEED_KI_FAN * cSPD_CTRL_PRD_FAN * cHZ_MAX_FAN / cA_MAX_FAN);
#else
	Motor_fan.para.spd.ki = (q31_t)(FIX_15 * cSPEED_KI_FAN * cSPD_CTRL_PRD_FAN * cHZ_MAX_FAN / cA_MAX_FAN);
#endif

#if defined(__SPEED_KP_RANGE)
	Motor_fan.para.spd.kp = (q31_t)(FIX_12 * cSPEED_KP_FAN * cHZ_MAX_FAN / cA_MAX_FAN);
#else
	Motor_fan.para.spd.kp = (q31_t)(FIX_15 * cSPEED_KP_FAN * cHZ_MAX_FAN / cA_MAX_FAN);
#endif

	/* current limit */
	Motor_fan.para.iq_lim = (q31_t)(FIX_31 * cIQ_LIM_FAN / cA_MAX_FAN);
	Motor_fan.para.id_lim = (q31_t)(FIX_31 * cID_LIM_FAN / cA_MAX_FAN);
	Motor_fan.para.err_ovc = (q15_t)(FIX_15 * cOVC_FAN / cA_MAX_FAN);

	/* Trigger compensation value setting */
	Motor_fan.para.TrgCompen = (int16_t)(FIX_15 * cTRG_COMPEN_FAN / cPWMPRD_FAN);
	/* Encoder parameter */
#if defined(__USE_ENCODER_FAN)
	Motor_fan.para.enc.pls2theta = ((FIX_32 / (cENC_PULSE_NUM_FAN * cENCMULTIPLE)) * (cPOLE_FAN / 2));
	Motor_fan.para.enc.deg_adjust = ELE_DEG32(cENC_DEG_ADJUST_FAN);
	Motor_fan.para.enc.plsnum = (cENC_PULSE_NUM_FAN * cENCMULTIPLE);
	Motor_fan.para.enc.pls2omega =
		(q15_t)(FIX_15 / ((cENC_PULSE_NUM_FAN * cENCMULTIPLE) / (cPOLE_FAN / 2)) / cCTRL_PRD_FAN / cHZ_MAX_FAN);
	Motor_fan.para.enc.ctrlprd = (uint32_t)(FIX_16 * cHZ_MAX_FAN * cCTRL_PRD_FAN);
#endif			  /* __USE_ENCODER_FAN */

#if !defined(__USE_VE_FAN)
	Motor_fan.para.PhCvMd = cPHCVMD_SVM;
#if (cSHUNT_TYPE_FAN == 1)
	Motor_fan.para.TrgPosMd = cTRGPOS_1SHUNT_SECOND_HALF;
#else
	Motor_fan.para.TrgPosMd = cTRGPOS_3SHUNT;
#endif

#endif  /* !defined(__USE_VE_FAN) */

#endif  /* defined(__CONTROL_MOTOR_FAN) */

	/***************************************************************************************************************************/

	_error->fan_i_hw_error.type = cError_FAN_I_HW;
	_error->fan_i_hw_error.priority = cFAN_I_HW_LEVEL;

	_error->fan_i_sw_error.type = cError_FAN_I_SW;
	_error->fan_i_sw_error.priority = cFAN_I_SW_LEVEL;

	_error->comp_i_hw_error.type = cError_COMP_I_HW;
	_error->comp_i_hw_error.priority = cCOMP_I_HW_LEVEL;

	_error->comp_i_sw_error.type = cError_COMP_I_SW;
	_error->comp_i_sw_error.priority = cCOMP_I_SW_LEVEL;

	_error->hpfc_i_hw_error.type = cError_HPFC_I_HW;
	_error->hpfc_i_hw_error.priority = cHPFC_I_HW_LEVEL;

	_error->hpfc_i_sw_error.type = cError_HPFC_I_SW;
	_error->hpfc_i_sw_error.priority = cHPFC_I_SW_LEVEL;

	_error->hpfc_over_vac_error.type = cError_HPFC_VAC_OVER;
	_error->hpfc_over_vac_error.priority = cHPFC_OVER_VAC_LEVEL;

	_error->hpfc_under_vac_error.type = cError_HPFC_VAC_UNDER;
	_error->hpfc_under_vac_error.priority = cHPFC_UNDER_VAC_LEVEL;

	_error->hpfc_over_vdc_error.type = cError_HPFC_VDC_OVER;;
	_error->hpfc_over_vdc_error.priority = cHPFC_OVER_VDC_LEVEL;

	_error->hpfc_under_vdc_error.type = cError_HPFC_VDC_UNDER;
	_error->hpfc_under_vdc_error.priority = cHPFC_UNDER_VDC_LEVEL;

	_error->hpfc_hz_error.type = cError_HPFC_HZ;
	_error->hpfc_hz_error.priority = cHPFC_HZ_LEVEL;

	_error->hpfc_zc_error.type = cError_HPFC_ZC;
	_error->hpfc_zc_error.priority = cHPFC_ZC_LEVEL;

	_error->ac_logical_temp_os_error.type = cError_Temp_OS;
	_error->ac_logical_temp_os_error.priority = cTEMP_OS_LEVEL;

	_error->ac_logical_temp_over_error.type = cError_Over_Temp;
	_error->ac_logical_temp_over_error.priority = cTEMP_OVER_LEVEL;

	_error->ac_logical_power_error.type = cError_Over_Power;
	_error->ac_logical_power_error.priority = cMOTOR_POWER_LEVEL;
}


/**
 *********************************************************************************************
 * @brief		Set MotorControl data for user.
 *
 * @return		none
 *********************************************************************************************
 */
void	B_User_MotorControl(void)
{
	vector_t *const _motor_comp = &Motor_comp;
	vector_t *const _motor_fan = &Motor_fan;
#if defined(__CONTROL_PFC)
	pfc_t *const _pfc = &Hpfc;
#endif

#if defined(__CONTROL_MOTOR_COMP)
	/***** user data setting for ch1 *****/

	/* Get EMG state */
	if (PMD_GetEMG_Status(TSB_PMD1) == cNormal)
	{
		_motor_comp->drv.state.flg.emg_H = CLEAR;
	}
	else
	{
		_motor_comp->drv.state.flg.emg_H = SET;
	}

#if defined(__USE_microE_COMMUNICATION_FUNC)
	_motor_comp->target_spd = target_spd_comp;
#endif /* __USE_microE_COMMUNICATION_FUNC */

	/* Set Speed */
#if (__TGTSPD_UNIT == 0)
	/* Hz of Electrical angle */
	_motor_comp->usr.omega_user.word = (q31_t)((q15_t)(_motor_comp->target_spd * FIX_15 / cHZ_MAX_COMP) << 16);
#elif (__TGTSPD_UNIT == 1)
	/* Hz of mechanical angle */
	_motor_comp->usr.omega_user.word = (q31_t)((q15_t)((_motor_comp->target_spd * cPOLEPAIR_COMP * FIX_15) / cHZ_MAX_COMP) << 16);
#elif (__TGTSPD_UNIT == 2)
	/* RPM of mechanical angle */
	_motor_comp->usr.omega_user.word = (q31_t)((q15_t)(((_motor_comp->target_spd * cPOLEPAIR_COMP * FIX_15) / 60) / cHZ_MAX_COMP) << 16);
#endif


	/* Motor ON/OFF */
	if (_motor_comp->target_spd > 0)
	{
		_motor_comp->usr.com_user.onoff = SET;
	}
	else
	{
		_motor_comp->usr.com_user.onoff = CLEAR;
		_motor_comp->drv.state.all = 0;									/* Clear error status. */

		if (PMD_GetEMG_Status(TSB_PMD1) == cEMGProtected)
		{
			PMD_ReleaseEMG_Protection(TSB_PMD1);
		}
	}
#endif /* __CONTROL_MOTOR_COMP */

#if defined(__CONTROL_MOTOR_FAN)
	/***** user data setting for ch0 *****/

	/* Get EMG state */
	if (PMD_GetEMG_Status(TSB_PMD0) == cNormal)
	{
		_motor_fan->drv.state.flg.emg_H = CLEAR;
	}
	else
	{
		_motor_fan->drv.state.flg.emg_H = SET;
	}

#if defined(__USE_microE_COMMUNICATION_FUNC)
	_motor_fan->target_spd = target_spd_fan;
#endif /* __USE_microE_COMMUNICATION_FUNC */

	/* Set Speed */
#if (__TGTSPD_UNIT == 0)
	/* Hz of Electrical angle */
	_motor_fan->usr.omega_user.word = (q31_t)((q15_t)(_motor_fan->target_spd * FIX_15 / cHZ_MAX_FAN) << 16);
#elif (__TGTSPD_UNIT == 1)
	/* Hz of mechanical angle */
	_motor_fan->usr.omega_user.word = (q31_t)((q15_t)((_motor_fan->target_spd * (cPOLE_FAN / 2) * FIX_15) / cHZ_MAX_FAN) << 16);
#elif (__TGTSPD_UNIT == 2)
	/* RPM of mechanical angle */
	_motor_fan->usr.omega_user.word = (q31_t)((q15_t)(((_motor_fan->target_spd * (cPOLE_FAN / 2) * FIX_15) / 60) / cHZ_MAX_FAN) << 16);
#endif

	/* Motor ON/OFF */
	if (_motor_fan->target_spd > 0)
	{
		_motor_fan->usr.com_user.onoff = SET;
	}
	else
	{
		_motor_fan->usr.com_user.onoff = CLEAR;
		_motor_fan->drv.state.all = 0;									/* Clear error status. */

		if (PMD_GetEMG_Status(TSB_PMD0) == cEMGProtected)
		{
			PMD_ReleaseEMG_Protection(TSB_PMD0);
		}
	}
#endif /* __CONTROL_MOTOR_FAN */

#if defined(__CONTROL_PFC)
	if (PMD_GetEMG_Status(TSB_PMD2) == cNormal)
	{
		_pfc->drv.state.flg.emg_H = CLEAR;
	}
	else
	{
		_pfc->drv.state.flg.emg_H = SET;
	}

	if (_pfc->drv.ac.power_mode == cPowerOFF)
	{
		_pfc->drv.state.all = 0;									/* Clear error status. */
		if (PMD_GetEMG_Status(TSB_PMD2) == cEMGProtected)
		{
			PMD_ReleaseEMG_Protection(TSB_PMD2);
		}
	}
#endif /* __CONTROL_PFC */
}

const uint16_t Temp_Exhaust_tbl[200] =			/* -49C to 150C */
{
	5, 5, 6, 6, 7, 7, 8, 8, 9, 10, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 21, 22, 24, 25,
	27, 29, 30, 32, 34, 37, 39, 41, 44, 47, 49, 52, 56, 59, 62, 66, 70, 74, 78, 82, 87, 91,
	96, 102, 107, 113, 119, 125, 131, 138, 145, 152, 160, 168, 176, 185, 193, 203, 212,
	222, 232, 243, 254, 265, 277, 289, 302, 315, 328, 342, 356, 371, 386, 401, 417, 434,
	451, 468, 486, 504, 523, 542, 562, 582, 603, 624, 645, 667, 690, 713, 736, 760, 784,
	809, 834, 859, 885, 911, 938, 965, 992, 1020, 1048, 1077, 1105, 1134, 1164, 1193,
	1223, 1253, 1284, 1314, 1345, 1376, 1407, 1438, 1470, 1501, 1533, 1564, 1596, 1628,
	1660, 1691, 1723, 1755, 1787, 1818, 1850, 1882, 1913, 1944, 1976, 2007, 2038, 2068,
	2099, 2129, 2159, 2189, 2219, 2249, 2278, 2307, 2336, 2364, 2392, 2420, 2448, 2475,
	2502, 2529, 2556, 2582, 2607, 2633, 2658, 2683, 2707, 2732, 2755, 2779, 2802, 2825,
	2847, 2869, 2891, 2913, 2934, 2955, 2975, 2995, 3015, 3034, 3053, 3072, 3091, 3109,
	3127, 3144, 3162, 3179, 3195, 3212, 3228, 3243, 3259, 3274, 3289, 3304, 3318, 3332,
	3346, 3360, 3373, 3386,
};

const uint16_t Temp_Room_tbl[200] =				/* -49C to 150C */
{
	58, 62, 66, 71, 75, 80, 86, 91, 97, 104, 110, 117, 125, 132, 140, 149, 158, 167, 177, 187,
	198, 209, 221, 233, 246, 259, 273, 288, 303, 318, 334, 351, 369, 387, 405, 425, 445, 465,
	487, 509, 531, 555, 579, 603, 629, 655, 681, 709, 737, 765, 795, 824, 855, 886, 918, 950,
	983, 1016, 1050, 1084, 1119, 1154, 1189, 1225, 1262, 1298, 1335, 1373, 1410, 1448, 1486,
	1524, 1562, 1600, 1638, 1677, 1715, 1754, 1792, 1830, 1868, 1906, 1944, 1982, 2020, 2057,
	2094, 2131, 2167, 2203, 2239, 2275, 2310, 2345, 2379, 2413, 2447, 2480, 2512, 2545, 2577,
	2608, 2639, 2669, 2699, 2728, 2757, 2786, 2814, 2841, 2868, 2895, 2921, 2946, 2971, 2996,
	3020, 3043, 3066, 3089, 3111, 3133, 3154, 3175, 3195, 3215, 3234, 3254, 3272, 3290, 3308,
	3326, 3343, 3359, 3375, 3391, 3407, 3422, 3437, 3451, 3465, 3479, 3493, 3506, 3519, 3531,
	3543, 3555, 3567, 3579, 3590, 3601, 3611, 3622, 3632, 3641, 3651, 3661, 3670, 3679, 3687,
	3696, 3704, 3712, 3720, 3728, 3736, 3743, 3750, 3757, 3764, 3771, 3778, 3784, 3790, 3796,
	3802, 3808, 3814, 3820, 3825, 3830, 3835, 3841, 3845, 3850, 3855, 3860, 3864, 3869, 3873,
	3877, 3881, 3885, 3889, 3893, 3897, 3901, 3904, 3908,
};

const uint16_t Temp_Pipe_tbl[200] =				/* -49C to 150C */
{
	22, 24, 26, 28, 30, 32, 35, 38, 41, 44, 47, 51, 55, 59, 63, 68, 73, 78, 83, 89, 96, 102, 109,
	117, 124, 133, 141, 150, 160, 170, 181, 192, 204, 217, 230, 243, 257, 272, 288, 304, 321,
	339, 358, 377, 397, 418, 440, 462, 485, 509, 534, 560, 587, 614, 642, 672, 701, 732, 764,
	796, 829, 863, 898, 934, 970, 1007, 1044, 1083, 1121, 1161, 1201, 1241, 1282, 1324, 1365,
	1407, 1450, 1493, 1536, 1579, 1622, 1666, 1709, 1753, 1796, 1840, 1883, 1927, 1970, 2013,
	2055, 2098, 2140, 2182, 2223, 2264, 2305, 2345, 2384, 2423, 2462, 2500, 2537, 2574, 2611,
	2646, 2682, 2716, 2750, 2783, 2816, 2848, 2879, 2910, 2940, 2969, 2998, 3026, 3054, 3081,
	3107, 3133, 3158, 3182, 3206, 3229, 3252, 3274, 3296, 3317, 3337, 3357, 3377, 3396, 3414,
	3432, 3449, 3466, 3483, 3499, 3515, 3530, 3545, 3559, 3573, 3587, 3600, 3613, 3625, 3637,
	3649, 3661, 3672, 3683, 3693, 3704, 3714, 3723, 3733, 3742, 3751, 3759, 3768, 3776, 3784,
	3792, 3799, 3807, 3814, 3821, 3827, 3834, 3840, 3846, 3852, 3858, 3864, 3869, 3875, 3880,
	3885, 3890, 3895, 3900, 3904, 3909, 3913, 3917, 3921, 3925, 3929, 3933, 3937, 3940, 3944,
	3947, 3950, 3954, 3957, 3960,
};

const uint16_t Temp_R25_B3435_5K_tbl[200] =		/* -49C to 150C@Pull-down 5.1K, 10K@25C */
{
	46, 49, 53, 56, 60, 64, 68, 73, 77, 82, 88, 93, 99, 105, 112, 118, 125, 133, 140, 149, 157, 166, 175, 185,
	195, 206, 217, 228, 240, 253, 266, 279, 293, 308, 323, 338, 355, 371, 389, 406, 425, 444, 464, 484, 505,
	526, 548, 571, 594, 618, 642, 667, 693, 719, 746, 774, 801, 830, 859, 889, 919, 949, 980, 1012, 1044,
	1076, 1109, 1142, 1176, 1209, 1244, 1278, 1313, 1348, 1383, 1419, 1455, 1490, 1526, 1562, 1599,
	1635, 1671, 1707, 1744, 1780, 1816, 1852, 1888, 1924, 1959, 1995, 2030, 2065, 2100, 2135, 2169,
	2203, 2237, 2270, 2303, 2336, 2369, 2401, 2432, 2464, 2495, 2525, 2555, 2585, 2614, 2643, 2672,
	2700, 2727, 2754, 2781, 2808, 2833, 2859, 2884, 2908, 2932, 2956, 2979, 3002, 3024, 3046, 3068,
	3089, 3110, 3130, 3150, 3169, 3189, 3207, 3226, 3244, 3261, 3278, 3295, 3312, 3328, 3344, 3359,
	3374, 3389, 3404, 3418, 3432, 3445, 3459, 3472, 3484, 3497, 3509, 3521, 3532, 3544, 3555, 3566,
	3576, 3587, 3597, 3607, 3617, 3626, 3635, 3644, 3653, 3662, 3670, 3679, 3687, 3695, 3703, 3710,
	3718, 3725, 3732, 3739, 3746, 3752, 3759, 3765, 3771, 3777, 3783, 3789, 3795, 3800, 3806, 3811,
	3816, 3821, 3826, 3831, 3836, 3841, 3845,
};

/**
*********************************************************************************************
* @brief		GetTempValue
*
* @param		uint16_t*_temp_tbl
* @param		int16_t _index_min
* @param		int16_t _index_max
* @param		uint16_t _ad_value
*
* @return	  ((m - offset) * 10 + x) (temperature[C])
*********************************************************************************************
*/
int16_t GetTempValue(const uint16_t* _temp_tbl, int16_t _index_min, int16_t _index_max, uint16_t _ad_value)
{
	uint16_t m, n, offset;

	if (_index_min < 0)
	{
		offset = 0 - _index_min;
	}
	else
	{
		offset = _index_min;
	}
	m = 0;
	n = _index_max - _index_min;
	if (_ad_value <= _temp_tbl[m])
		return (_index_min * 10);						/* Unit: 0.1C */
	if (_ad_value >= _temp_tbl[n])
		return (_index_max * 10);						/* Unit: 0.1C */

	while (1)
	{
		if (_ad_value >= _temp_tbl[((m + n) / 2)])
		{
			if (m == (m + n) / 2)
			{
				if (_ad_value >= ((_temp_tbl[m] + _temp_tbl[(m + 1)]) >> 1))
				{
					return ((m - offset) * 10 + 5);		/* Unit: 0.1C,  Get 0.5C with 1.0C table */
				}
				else
				{
					return ((m - offset) * 10);			/* Unit: 0.1C */
				}
			}
			m = (m + n) / 2;
		}
		else
		{
			n = (m + n) / 2;
		}
	}
}

/**
*********************************************************************************************
* @brief		Binary search
*
* @param		uint16_t*_temp_tbl
* @param		int16_t _index_min
* @param		int16_t _index_max
* @param		uint16_t _value
*
* @return
*********************************************************************************************
*/
int16_t Binary_search(const int16_t * _temp_tbl, int16_t _index_min, int16_t _index_max, int16_t _value)
{
	uint16_t m, n;
	m = 0;
	n = _index_max - _index_min;
	if (_value <= _temp_tbl[m])
		return (_index_min);
	if (_value >= _temp_tbl[n])
		return (_index_max );

	while (1)
	{
		if (_value >= _temp_tbl[((m + n) / 2)])
		{
			if (m == (m + n) / 2)
			{
				if (_value >= ((_temp_tbl[m] + _temp_tbl[(m + 1)]) >> 1))
				{
					return (m + 1);
				}
				else
				{
					return (m );
				}
			}
			m = (m + n) / 2;
		}
		else
		{
			n = (m + n) / 2;
		}
	}
}

/**
*********************************************************************************************
* @brief	   B_Protect_TemperatureMonitorInit
*
* @param	   none
*
* @return	  none
*********************************************************************************************
*/
void B_Protect_TemperatureMonitorInit()
{
	temperature_t *const _temp = &Temp;

	/* AD register initial for _temp check */
	ADC_SetSWTrg(ADx0, ADCx_REG0, ADCx_AIN0);							/* M4KLA: HVMOS---AINC2-REG8 */
	ADC_SetSWTrg(ADx1, ADCx_REG1, ADCx_AIN1);							/* M4KLA: Outroom---AINB0-REG0 */
	ADC_SetSWTrg(ADx2, ADCx_REG2, ADCx_AIN2);							/* M4KLA: Outpipe--AINB2-REG1 */
	ADC_SetSWTrg(ADx3, ADCx_REG3, ADCx_AIN3);							/* M4KLA: EXH---AINB1-REG2 */
	ADC_SetSWTrg(ADx4, ADCx_REG4, ADCx_AIN4);							/* M4KLA: Diode---AINA13-REG8 */
	ADC_SetSWTrg(ADx5, ADCx_REG5, ADCx_AIN5);							/* M4KLA: IGBT---AINA18-REG9 */

	/* Enable SW trigger */
	ADC_Start(TSB_ADC, ADC_TRG_SINGLE);
	ADC_Start(TSB_ADB, ADC_TRG_SINGLE);
	ADC_Start(TSB_ADA, ADC_TRG_SINGLE);
	/* T para initial */
	/* EXH 110stop */
	_temp->t_exh_protect.tc_sl_up = cEXH_SL_UP * cTIMES;				/* [C] EXH slow up speed temperature */
	_temp->t_exh_protect.tc_no_up = cEXH_NO_UP * cTIMES;				/* [C] EXH no increase temperature */
	_temp->t_exh_protect.tc_sl_down = cEXH_SL_DW * cTIMES;				/* [C] EXH slow down speed temperature */
	_temp->t_exh_protect.tc_over_emg = cEXH_OVER_EMG * cTIMES;			/* [C] EXH EMG temperature */

	/* HVMOS 90stop */
	_temp->t_hvmos_protect.tc_sl_up = cHVMOS_SL_UP * cTIMES;			/* [C] HVMOS slow up speed temperature */
	_temp->t_hvmos_protect.tc_no_up = cHVMOS_NO_UP * cTIMES;			/* [C] HVMOS no increase speed temperature */
	_temp->t_hvmos_protect.tc_sl_down = cHVMOS_SL_DW * cTIMES;			/* [C] HVMOS slow down speed temperature */
	_temp->t_hvmos_protect.tc_over_emg = cHVMOS_OVER_EMG * cTIMES;		/* [C] HVMOS EMG temperature */

	/* IGBT 90stop */
	_temp->t_igbt_protect.tc_sl_up = cIGBT_SL_UP * cTIMES;				/* [C] IGBT slow up speed temperature */
	_temp->t_igbt_protect.tc_no_up = cIGBT_NO_UP * cTIMES;				/* [C] IGBT no increase speed temperature */
	_temp->t_igbt_protect.tc_sl_down = cIGBT_SL_DW * cTIMES;			/* [C] IGBT slow down speed temperature */
	_temp->t_igbt_protect.tc_over_emg = cIGBT_OVER_EMG * cTIMES;		/* [C] IGBT EMG temperature */

	_temp->t_diode_protect.tc_sl_up = cDIODE_SL_UP * cTIMES;			/* [C] DIODE slow up speed temperature */
	_temp->t_diode_protect.tc_no_up = cDIODE_NO_UP * cTIMES;			/* [C] DIODE no increase speed temperature */
	_temp->t_diode_protect.tc_sl_down = cDIODE_SL_DW * cTIMES;			/* [C] DIODE slow down speed temperature */
	_temp->t_diode_protect.tc_over_emg = cDIODE_OVER_EMG * cTIMES;		/* [C] DIODE EMG temperature */

	_temp->T_outpipe_protect.tc_sl_up = cOUTPIPE_SL_UP * cTIMES;		/* [C] Outpipe slow up speed temperatur */
	_temp->T_outpipe_protect.tc_no_up = cOUTPIPE_NO_UP * cTIMES;		/* [C] Outpipe no increase speed temper */
	_temp->T_outpipe_protect.tc_sl_down = cOUTPIPE_SL_DW * cTIMES;		/* [C] Outpipe slow down speed temper */
	_temp->T_outpipe_protect.tc_over_emg = cOUTPIPE_OVER_EMG * cTIMES;	/* [C] Outpipe EMG temperature */

	_temp->omega_fl2.word = (q31_t)((q15_t)(cOMEGA_DW_LIMIT * FIX_15 / cHZ_MAX_COMP) << 16);  /* limit speed */
}

/**
*********************************************************************************************
* @brief	   B_Protect_Temperature
*
* @param	   temperature_t* const _temp		  structure for temperature protect function
* @param	   vector_t* const _motor			  structure for motor compressor
*
* @return	  none
*********************************************************************************************
*/
void B_Protect_Temperature(temperature_t * const _temp, vector_t* const _motor)
{
	/* get temperature */
	B_Protect_GetTemperature(_temp);
	B_Protect_CheckTemperatureError(_temp);

	/* get motor control status by temperature */
	B_Protect_TemperatureStatusSel(&(_temp->t_exh_protect), _temp->t_exh, cTEMP_CNT);
	B_Protect_TemperatureStatusSel(&(_temp->t_hvmos_protect), _temp->t_hvmos, cTEMP_CNT);
	B_Protect_TemperatureStatusSel(&(_temp->T_outpipe_protect), _temp->t_outpipe, cTEMP_CNT);
	B_Protect_TemperatureStatusSel(&(_temp->t_igbt_protect), _temp->t_igbt, cTEMP_CNT);
	B_Protect_TemperatureStatusSel(&(_temp->t_diode_protect), _temp->t_diode, cTEMP_CNT);

	/* status selection */
	_temp->t_st = (temp_protect_st_e)(((_temp->t_exh_protect.user_st) > (_temp->t_hvmos_protect.user_st))? (_temp->t_exh_protect.user_st) : (_temp->t_hvmos_protect.user_st));
	_temp->t_st = (temp_protect_st_e)((_temp->t_st > _temp->T_outpipe_protect.user_st)? (_temp->t_st) : (_temp->T_outpipe_protect.user_st));
	_temp->t_st = (temp_protect_st_e)((_temp->t_st > _temp->t_igbt_protect.user_st)? (_temp->t_st) : (_temp->t_igbt_protect.user_st));
	_temp->t_st = (temp_protect_st_e)((_temp->t_st > _temp->t_diode_protect.user_st)? (_temp->t_st) : (_temp->t_diode_protect.user_st));
	_temp->t_st = (temp_protect_st_e)( (_temp->t_st > _temp->t_hvmos_protect.user_st)? (_temp->t_st) : (_temp->t_hvmos_protect.user_st));

	if (_temp->t_st == cTEMP_SLOW_DECREASE)
	{
		_temp->t_index.all = 0;
		if (_temp->t_exh_protect.user_st == cTEMP_SLOW_DECREASE)
		{
			_temp->t_index.flg.t_exh = 1;
		}
		else if (_temp->t_hvmos_protect.user_st == cTEMP_SLOW_DECREASE)
		{
			_temp->t_index.flg.t_hvmos = 1;
		}
		else if (_temp->T_outpipe_protect.user_st == cTEMP_SLOW_DECREASE)
		{
			_temp->t_index.flg.t_outdoor_pipe = 1;
		}
		else if (_temp->t_igbt_protect.user_st == cTEMP_SLOW_DECREASE)
		{
			_temp->t_index.flg.t_igbt = 1;
		}
		else if (_temp->t_diode_protect.user_st == cTEMP_SLOW_DECREASE)
		{
			_temp->t_index.flg.t_diode = 1;
		}
	}
	else
	{
		_temp->t_index.all = 0;
		if (_temp->t_exh_protect.user_st == cTEMP_SLOW_INCREASE)
		{
			_temp->t_index.flg.t_exh = 1;
		}
		else if (_temp->t_hvmos_protect.user_st == cTEMP_SLOW_INCREASE)
		{
			_temp->t_index.flg.t_hvmos = 1;
		}
		else if (_temp->T_outpipe_protect.user_st == cTEMP_SLOW_INCREASE)
		{
			_temp->t_index.flg.t_outdoor_pipe = 1;
		}
		else if (_temp->t_igbt_protect.user_st == cTEMP_SLOW_INCREASE)
		{
			_temp->t_index.flg.t_igbt = 1;
		}
		else if (_temp->t_diode_protect.user_st == cTEMP_SLOW_INCREASE)
		{
			_temp->t_index.flg.t_diode = 1;
		}
	}
}

/**
*********************************************************************************************
* @brief	   B_Protect_GetTemperature
*
* @param	   temperature_t* const _temp	structure for temperature protect
*
* @return	  none
*********************************************************************************************
*/
void B_Protect_GetTemperature(temperature_t * const _temp)
{
	if (ST_SNGF0 == 0)			/* exhaust,room,pipe */
	{
		/* get outRoom temperature */
		_temp->R_OUTDOORT.All = OUTDOORT_P;			/* TSB_ADC->REG8; */
		_temp->r_outdoort_f += (_temp->R_OUTDOORT.Bit.ADResult - _temp->r_outdoort_f) >> 6;
		_temp->t_outdoor = GetTempValue(Temp_Room_tbl, cTEMP_DW1, cTEMP_UP1, _temp->r_outdoort_f);

		/* get outpipe temperature */
		_temp->R_OUTPIPE.All = PIPE_P;				/* TSB_ADC->REG9 */
		_temp->r_outpipe_f += (_temp->R_OUTPIPE.Bit.ADResult - _temp->r_outpipe_f) >> 6;
		_temp->t_outpipe = GetTempValue(Temp_Pipe_tbl, cTEMP_DW1, cTEMP_UP1, _temp->r_outpipe_f);

		/* get outEXHAUST temperature */
		_temp->R_EXHAUST.All = EXH_P;				/* TSB_ADC->REG10; */
		_temp->r_exhaust_f += (_temp->R_EXHAUST.Bit.ADResult - _temp->r_exhaust_f) >> 6;
		_temp->t_exh = GetTempValue(Temp_Exhaust_tbl, cTEMP_DW1, cTEMP_UP1, _temp->r_exhaust_f);
	}

	if (ST_SNGF1 == 0)
	{
		/* get HVMOS temperature */
		_temp->R_HVMOS.All = HVMOS_P;				/* TSB_ADB->REG0; */
		_temp->r_hvmos_f += (_temp->R_HVMOS.Bit.ADResult - _temp->r_hvmos_f) >> 6;
		_temp->t_hvmos = GetTempValue(Temp_R25_B3435_5K_tbl, cTEMP_DW1, cTEMP_UP1, _temp->r_hvmos_f);
	}

	if (ST_SNGF2 == 0)
	{
		/* get IGBT  temperature */
		_temp->R_IGBT.All = IGBT_P;					/* TSB_ADA->REG9; */
		_temp->r_igbt_f += (_temp->R_IGBT.Bit.ADResult - _temp->r_igbt_f) >> 6;
		_temp->t_igbt = GetTempValue(Temp_R25_B3435_5K_tbl, cTEMP_DW1, cTEMP_UP1, _temp->r_igbt_f);

		/* get DIODE  temperature */
		_temp->R_DIODE.All = DIODE_P;				/* TSB_ADA->REG9; */
		_temp->r_diode_f += (_temp->R_DIODE.Bit.ADResult - _temp->r_diode_f) >> 6;
		_temp->t_diode = GetTempValue(Temp_R25_B3435_5K_tbl, cTEMP_DW1, cTEMP_UP1, _temp->r_diode_f);
	}

	ADC_Start(TSB_ADA, ADC_TRG_SINGLE);
	ADC_Start(TSB_ADB, ADC_TRG_SINGLE);
	ADC_Start(TSB_ADC, ADC_TRG_SINGLE);
}

/**
*********************************************************************************************
* @brief	   B_Protect_CheckTemperatureError
*
* @param	   temperature_t* const  _temp  structure for temperature protect
*
* @return	  none
*********************************************************************************************
*/
void B_Protect_CheckTemperatureError(temperature_t * const _temp)
{
	vector_t *_motor = &Motor_comp;

	if (_temp->r_outdoort_f < cOPEN)
	{
		if (++_temp->cnt0 > cOPEN_SHORT_CNT)
		{
#if defined(__USE_ALL_TEMP_ERROR)
			_temp->state.flg.f_open_outdoor = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
#endif
		}
	}
	else
	{
		_temp->cnt0 = 0;
	}

	if (_temp->r_outpipe_f < cOPEN)
	{
		if (++_temp->cnt1 > cOPEN_SHORT_CNT)
		{
#if defined(__USE_ALL_TEMP_ERROR)
			_temp->state.flg.f_open_pipe = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
#endif
		}
	}
	else
	{
		_temp->cnt1 = 0;
	}

	if (_temp->r_exhaust_f < cOPEN)
	{
		if (++_temp->cnt2 > cOPEN_SHORT_CNT)
		{
#if defined(__USE_ALL_TEMP_ERROR)
			_temp->state.flg.f_open_exhaust = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
#endif
		}
	}
	else
	{
		_temp->cnt2 = 0;
	}

	if (_temp->r_hvmos_f < cOPEN)
	{
		if (++_temp->cnt3 > cOPEN_SHORT_CNT)
		{
			_temp->state.flg.f_open_hvmos = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
		}
	}
	else
	{
		_temp->cnt3 = 0;
	}

	if (_temp->r_outdoort_f > cSHORTC)
	{
		if (++_temp->cnt4 > cOPEN_SHORT_CNT)
		{
#if defined(__USE_ALL_TEMP_ERROR)
			_temp->state.flg.f_short_outdoor = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
#endif
		}
	}
	else
	{
		_temp->cnt4 = 0;
	}

	if (_temp->r_outpipe_f > cSHORTC)
	{
		if (++_temp->cnt5 > cOPEN_SHORT_CNT)
		{
#if defined(__USE_ALL_TEMP_ERROR)
			_temp->state.flg.f_short_pipe = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
#endif
		}
	}
	else
	{
		_temp->cnt5 = 0;
	}

	if (_temp->r_exhaust_f > cSHORTC)
	{
		if (++_temp->cnt6 > cOPEN_SHORT_CNT)
		{
#if defined(__USE_ALL_TEMP_ERROR)
			_temp->state.flg.f_short_exhaust = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
#endif
		}
	}
	else
	{
		_temp->cnt6 = 0;
	}

	if (_temp->r_hvmos_f > cSHORTC)
	{
		if (++_temp->cnt7 > cOPEN_SHORT_CNT)
		{
			_temp->state.flg.f_short_hvmos = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
		}
	}
	else
	{
		_temp->cnt7 = 0;
	}

	if (_temp->r_igbt_f < cOPEN)
	{
		if (++_temp->cnt8 > cOPEN_SHORT_CNT)
		{
			_temp->state.flg.f_open_igbt = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
		}
	}
	else
	{
		_temp->cnt8 = 0;
	}

	if (_temp->r_igbt_f > cSHORTC)
	{
		if (++_temp->cnt9 > cOPEN_SHORT_CNT)
		{
			_temp->state.flg.f_short_igbt = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
		}
	}
	else
	{
		_temp->cnt9 = 0;
	}

	if (_temp->r_diode_f < cOPEN)
	{
		if (++_temp->cnt10 > cOPEN_SHORT_CNT)
		{
			_temp->state.flg.f_open_diode = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
		}
	}
	else
	{
		_temp->cnt10 = 0;
	}

	if (_temp->r_diode_f > cSHORTC)
	{
		if (++_temp->cnt11 > cOPEN_SHORT_CNT)
		{
			_temp->state.flg.f_short_diode = 1;
			_motor->temp_t.state.flg.emg_temp_os = SET;
		}
	}
	else
	{
		_temp->cnt11 = 0;
	}
}

/**
*********************************************************************************************
* @brief	   B_Protect_TemperatureStatusSel
*
* @param	   tc_Lim_t* _tc
* @param		int16_t  _t
*
* @return	  none
*********************************************************************************************
*/
void B_Protect_TemperatureStatusSel(tc_Lim_t * _tc, int16_t _t, uint16_t times)
{
	vector_t *_motor_comp = &Motor_comp;

	if (_t > _tc->tc_over_emg)
	{
		if (++_tc->t_c > times)
		{
			_tc->t_over_emg = 1;
			_motor_comp->temp_t.state.flg.emg_temp_over = SET;
			_tc->t_c = 0;
		}
	}
	else
	{
		_tc->t_c = 0;
	}
}

/*
 *********************************************************************************************
 * @brief	   B_Error_Collection
 *
* @param   None
 * @return	  none
 *********************************************************************************************
 */
void B_Error_Collection(void)
{
	vector_t *_motor_fan = &Motor_fan;
	vector_t *_motor_comp = &Motor_comp;
	error_t *_error = &Error;
	pfc_t *_hpfc = &Hpfc;
	
	/* Fan */
	if (_motor_fan->drv.state.all != 0)
	{
		if (_motor_fan->drv.state.flg.emg_I != 0)
		{
			_error->fan.error_type_buf[_error->fan.pt++] = _error->fan_i_hw_error;
		}
		if (_motor_fan->drv.state.flg.emg_S != 0)
		{
			_error->fan.error_type_buf[_error->fan.pt++] = _error->fan_i_sw_error;
		}

		uint8_t i, j;
		error_type_t temp;
		for (i = 0; i < _error->fan.pt; i++)
		{
			for (j = i + 1; j < _error->fan.pt; j++)
			{
				if (_error->fan.error_type_buf[i].priority > _error->fan.error_type_buf[j].priority)
				{
					temp = _error->fan.error_type_buf[i];
					_error->fan.error_type_buf[i] = _error->fan.error_type_buf[j];
					_error->fan.error_type_buf[j] = temp;
				}
			}
		}
		_error->fan.pt = 0;
	}

	/* Comp */
	if (_motor_comp->drv.state.all != 0)
	{
		if (_motor_comp->drv.state.flg.emg_I != 0)
		{
			_error->comp.error_type_buf[_error->comp.pt++] =  _error->comp_i_hw_error;
		}
		if (_motor_comp->drv.state.flg.emg_S != 0)
		{
			_error->comp.error_type_buf[_error->comp.pt++] = _error->comp_i_sw_error;
		}

		uint8_t i, j;
		error_type_t temp;
		for (i = 0; i < _error->comp.pt; i++)
		{
			for (j = i + 1; j < _error->comp.pt; j++)
			{
				if (_error->comp.error_type_buf[i].priority > _error->comp.error_type_buf[j].priority)
				{
					temp = _error->comp.error_type_buf[i];
					_error->comp.error_type_buf[i] = _error->comp.error_type_buf[j];
					_error->comp.error_type_buf[j] = temp;
				}
			}
		}
		_error->comp.pt = 0;
	}

	/* Hpfc */
	if (_hpfc->drv.state.all != 0)
	{
		if (_hpfc->drv.state.flg.emg_H != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] =  _error->hpfc_i_hw_error;
		}
		if (_hpfc->drv.state.flg.emg_S != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] = _error->hpfc_i_sw_error;
		}
		if (_hpfc->drv.state.flg.over_Vac != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] = _error->hpfc_over_vac_error;
		}
		if (_hpfc->drv.state.flg.under_Vac != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] = _error->hpfc_under_vac_error;
		}
		if (_hpfc->drv.state.flg.over_Vdc != 0 || _hpfc->drv.state.flg.emg_DC != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] = _error->hpfc_over_vdc_error;
		}
		if (_hpfc->drv.state.flg.under_Vdc != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] = _error->hpfc_under_vdc_error;
		}
		if (_hpfc->drv.state.flg.emg_Hz != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] = _error->hpfc_hz_error;
		}
		if (_hpfc->drv.state.flg.emg_ZC != 0)
		{
			_error->hpfc.error_type_buf[_error->hpfc.pt++] = _error->hpfc_zc_error;
		}

		uint8_t i, j;
		error_type_t temp;
		for (i = 0; i < _error->hpfc.pt; i++)
		{
			for (j = i + 1; j < _error->hpfc.pt; j++)
			{
				if (_error->hpfc.error_type_buf[i].priority > _error->hpfc.error_type_buf[j].priority)
				{
					temp = _error->hpfc.error_type_buf[i];
					_error->hpfc.error_type_buf[i] = _error->hpfc.error_type_buf[j];
					_error->hpfc.error_type_buf[j] = temp;
				}
			}
		}
		_error->hpfc.pt = 0;
	}

	if (_motor_comp->temp_t.state.all != 0)
	{
		if (_motor_comp->temp_t.state.flg.emg_temp_os != 0)
		{
			_error->ac_logical.error_type_buf[_error->ac_logical.pt++] = _error->ac_logical_temp_os_error;
		}
		if (_motor_comp->temp_t.state.flg.emg_temp_over != 0)
		{
			_error->ac_logical.error_type_buf[_error->ac_logical.pt++] = _error->ac_logical_temp_over_error;
		}

		uint8_t i, j;
		error_type_t temp;
		for (i = 0 ; i < _error->ac_logical.pt; i++)
		{
			for (j = i + 1; j < _error->ac_logical.pt; j++)
			{
				if (_error->ac_logical.error_type_buf[i].priority > _error->ac_logical.error_type_buf[j].priority)
				{
					temp = _error->ac_logical.error_type_buf[i];
					_error->ac_logical.error_type_buf[i] = _error->ac_logical.error_type_buf[j];
					_error->ac_logical.error_type_buf[j] = temp;
				}
			}
		}
		_error->ac_logical.pt = 0;
	}
}
/**
 *********************************************************************************************
 * @brief	   B_Error_DisplayCtrl
 *
  *@param	   none
 *
 * @return  none
 *
 *********************************************************************************************
 */
void B_Error_DisplayCtrl(void)
{
	error_t *_error = &Error;

	B_Error_Collection();

	switch (_error->hpfc.error_type_buf[0].type)
	{
		case cError_None_Err:	   /* No error */
			Error_led.times[0] = 0;
			Error_led.blink_cnt[0] = c500MS;
			Error_led.interval_cnt[0] = 0;
			LED_Disp[0].LED = 1;
			LED4 = 1;
			break;

		case cError_HPFC_I_HW:	  /* PFC H/W over current EMG2 error */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 1;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_HPFC_I_SW:	  /* PFC S/W Over Iac */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 2;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_HPFC_VAC_OVER:  /* PFC S/W Over Vac */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 3;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_HPFC_VAC_UNDER: /* PFC S/W Under Vac */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 4;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_HPFC_VDC_OVER:  /* PFC S/W Over Vdc */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 5;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_HPFC_VDC_UNDER: /* PFC S/W Under Vdc */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 6;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_HPFC_HZ:		/* PFC S/W Under Vdc */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 7;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_HPFC_ZC:		/* PFC S/W Under Vdc */
			Error_led.num = 0;
			LED_Disp[Error_led.num].sLED = 4;
			LED_Disp[Error_led.num].time = 8;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		default:
			Error_led.times[0] = 0;
			Error_led.blink_cnt[0] = c500MS;
			Error_led.interval_cnt[0] = 0;
			LED4 = 1;
			break;
	}

	switch (_error->fan.error_type_buf[0].type )
	{
		case cError_None_Err:	   /* No error */
			Error_led.times[1] = 0;
			Error_led.blink_cnt[1] = c500MS;
			Error_led.interval_cnt[1] = 0;
			LED_Disp[1].LED = 1;
			LED2 = 1;
			break;

		case cError_FAN_I_HW:	   /* Fan H/W over current EMG0 error */
			Error_led.num = 1;
			LED_Disp[Error_led.num].sLED = 2;
			LED_Disp[Error_led.num].time = 1;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_FAN_I_SW:	   /* Fan S/W Over current */
			Error_led.num = 1;
			LED_Disp[Error_led.num].sLED = 2;
			LED_Disp[Error_led.num].time = 2;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		default:
			Error_led.times[1] = 0;
			Error_led.blink_cnt[1] = c500MS;
			Error_led.interval_cnt[1] = 0;
			LED2 = 1;
			break;
	}

	switch (_error->comp.error_type_buf[0].type )
	{
		case cError_None_Err:	   /* No error */
			Error_led.times[2] = 0;
			Error_led.blink_cnt[2] = c500MS;
			Error_led.interval_cnt[2] = 0;
			LED_Disp[2].LED = 1;
			LED3 = 1;
			break;

		case cError_COMP_I_HW:	  /* Compressor H/W over current EMG1 error */
			Error_led.num = 2;
			LED_Disp[Error_led.num].sLED = 3;
			LED_Disp[Error_led.num].time = 1;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_COMP_I_SW:	  /* Compressor S/W Over current */
			Error_led.num = 2;
			LED_Disp[Error_led.num].sLED = 3;
			LED_Disp[Error_led.num].time = 2;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		default:
			Error_led.times[2] = 0;
			Error_led.blink_cnt[2] = c500MS;
			Error_led.interval_cnt[2] = 0;
			LED3 = 1;
			break;
	}

	switch (_error->ac_logical.error_type_buf[0].type )
	{
		case cError_None:		   /* No error */
			Error_led.times[3] = 0;
			Error_led.blink_cnt[3] = c500MS;
			Error_led.interval_cnt[3] = 0;
			LED_Disp[3].LED = 1;
			LED1 = 1;
			break;

		case cError_Temp_OS:		/* Temperarure sensor open/short error*/
			Error_led.num = 3;
			LED_Disp[Error_led.num].sLED = 1;
			LED_Disp[Error_led.num].time = 1;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		case cError_Over_Temp:	  /* Over Temperarure error(indoor pipe,outdoor pipe,exhaust,HVMOS)*/
			Error_led.num = 3;
			LED_Disp[Error_led.num].sLED = 1;
			LED_Disp[Error_led.num].time = 2;
			B_Error_Display(LED_Disp[Error_led.num].time, LED_Disp[Error_led.num].sLED, Error_led.num);
			break;

		default:
			Error_led.times[3] = 0;
			Error_led.blink_cnt[3] = c500MS;
			Error_led.interval_cnt[3] = 0;
			LED1 = 1;
			break;
	}

	if (Error_led.cnt == 3)
	{
		Error_led.cnt = 0;
	}
}

/**
 *********************************************************************************************
 * @brief	   B_Error_Display
 *
 *@param	  uint8_t _time
 *@param	 uint8_t _sLED
 *@param	uint8_t i
 *
 * @return	  none
 *
 *********************************************************************************************
 */
void B_Error_Display(uint8_t _time, uint8_t _sLED, uint8_t i)
{
	if (Error_led.times[i] < (2 * _time))
	{
		Error_led.blink_cnt[i]++;
		if (Error_led.blink_cnt[i] >= c500MS)
		{
			Error_led.blink_cnt[i] = 0;
			if (Error_led.times[i] < (2 * _time))
			{
				LED_Disp[i].LED = ~LED_Disp[i].LED;
			}
			Error_led.times[i]++;
		}
	}

	if (Error_led.times[i] >= (2 * _time))
	{
		LED_Disp[i].LED = 1;		/* LED OFF */
		Error_led.interval_cnt[i]++;

		if (Error_led.interval_cnt[i] >= c2000MS)
		{
			LED_Disp[i].LED = 0;	/* LED ON */
			Error_led.times[i] = 0;
			Error_led.blink_cnt[i] = c500MS;
			LED_Disp[i].LED = 1;
			Error_led.interval_cnt[i] = 0;
		}
	}

	/* Select LED */
	switch (_sLED)
	{
		case 1:
			LED1 = LED_Disp[i].LED;
			break;
		case 2:
			LED2 = LED_Disp[i].LED;
			break;
		case 3:
			LED3 = LED_Disp[i].LED;
			break;
		case 4:
			LED4 = LED_Disp[i].LED;
			break;
		default:
			break;
	}
}

/*********************************** END OF FILE ******************************/
